package com.jingdong.common.utils.text

import android.content.Context
import android.os.Handler
import android.os.Looper
import com.jingdong.common.utils.CommonUtil
import com.jingdong.common.utils.text.ScaleModeConstants.TEXT_SCALE_MODE_KEY
import com.jingdong.common.utils.text.ScaleModeConstants.TEXT_SCALE_MODE_STANDARD

/**
 * 文本缩放设置助手类，单例实现
 * 提供API能力说明：
 * 1.注册/解绑指定文本字号变更监听器
 * 2.获取字号缩放模式
 *  2.1获取字号缩放模式使用场景：有些UI有相对关系，不能通过直接修改字号实现，需要动态调整布局时需要使用此模式。
 *
 * @author wangguowei1
 * @date 2019-10-09
 */
open class TextScaleModeHelper {

    companion object {
        /**
         * 创建单例对象
         */
        val instance = SingletonHolder.holder
        /**
         * mainThreadHandler
         */
        @Volatile
        private var mainThreadHandler: Handler? = null
    }

    /**
     * 记录最近一次缩放模式
     */
    private var lastTextScaleMode: String? = null

    private constructor() {
        mainThreadHandler = Handler(Looper.getMainLooper())
    }

    /**
     * 静态内部类单例实现
     */
    private object SingletonHolder {
        val holder = TextScaleModeHelper()
    }

    /**
     * 初始化字号变更观察者
     */
    private val textSizeObserver: TextSizeObserver by lazy {
        TextSizeObserver()
    }

    /**
     * 注册指定监听器
     *
     * @param listener
     */
    fun addOnTextSizeChangeListener(listener: OnTextScaleModeChangeListener) {
        textSizeObserver.addOnTextSizeChangeListener(listener)
    }

    /**
     * 移除对应监听器
     *
     * @param listener
     */
    fun removeOnTextSizeChangeListener(listener: OnTextScaleModeChangeListener) {
        textSizeObserver.removeOnTextSizeChangeListener(listener)
    }

    /**
     * 分发文本缩放模式变更通知
     * 说明：通过 handler 在主线程执行通知分发
     */
    internal fun dispatchOnTextSizeChange() {
        mainThreadHandler?.post { textSizeObserver.dispatchOnTextSizeChange() }
    }

    /**
     * 获取文本缩放模式
     */
    fun getTextSizeScaleMode(): String {
        if (lastTextScaleMode.isNullOrEmpty()) {
            lastTextScaleMode = CommonUtil.getJdSharedPreferences().getString(TEXT_SCALE_MODE_KEY, TEXT_SCALE_MODE_STANDARD)
        }
        return lastTextScaleMode!!
    }

    /**
     * 获取安档位缩放所对应的字号
     *
     * @param context 上下文引用
     * @param originalSP 原始字号，单位SP 示例：UI标注字号20px,originalSP入参：10 / 2 sp
     * @return 依据缩放档位计算出对应的字号，单位：SP
     */
    fun getScaleSize(context: Context, originalSP: Float): Float {
        return TextScaleSizeHelper.dynamicFontSize(context, originalSP * 2) / 2
    }

    /**
     * 更新最近一次文本缩放模式
     */
    internal fun setLastTextScaleMode(currentTextScaleMode: String) {
        lastTextScaleMode = currentTextScaleMode
    }
}