--[[--This module is responsible for generating the quickstart guide.
]]
local DataStorage = require("datastorage")
local FileConverter = require("apps/filemanager/filemanagerconverter")
local DocSettings = require("docsettings")
local Version = require("version")
local FFIUtil = require("ffi/util")
local T = FFIUtil.template
local lfs = require("libs/libkoreader-lfs")
local _ = require("gettext")

local QuickStart = {
    quickstart_force_show_version = 201511982,
}

local language = G_reader_settings:readSetting("language") or "en"
local version = Version:getNormalizedCurrentVersion()
local rev = Version:getCurrentRevision()

local quickstart_guide = T(_([[
# KOReader Quickstart Guide

Welcome to KOReader.

You can activate the menu by swiping down from the top of the screen. Clicking outside the menu or swiping up on the menu will discard it.

Turning pages can be done either by swiping left and right or by single taps on the left or right side of the screen.

### Contents

* [Menu](#menu)
  * [Main menu](#main-menu)
  * [Settings](#settings)
* [File browser](#file-browser)


## Menu <a id="menu"></a>

Most menu items will show more information about what they do by long pressing on the item.

### Main <a id="main-menu"></a>

![Menu](../resources/icons/mdlight/appbar.menu.svg) You can always view this quickstart guide again through *Help* → *Quickstart guide* in the top right menu.

### Settings <a id="settings"></a>

![Settings](../resources/icons/mdlight/appbar.settings.svg) You can change the language and other settings through the gear icon.


## File browser <a id="file-browser"></a>

The file browser will only show document or ebook files that KOReader can read.

In the file browser, you can tap on any file to open it. Long press on any file to bring up a menu with more options. The location path display above the list of files and folders shows you which folder you're viewing. The `../` entry, at the top of the listed folders, lets you go *up* one level. For instance, if you are at `/mnt/onboard` now, tapping the `../` will bring you to `/mnt/`.

Once you have found the folder you have your books listed in, you can long press the selection that opens that folder and you should see a message box popup with the option to **Set as HOME folder**.

## Defaults <a id="defaults"></a>

If you long press an option KOReader will prompt you to set it as the global default.

## Wiki <a id="wiki"></a>

Please refer to the wiki at <a href="https://github.com/koreader/koreader/wiki">https://github.com/koreader/koreader/wiki</a> for more documentation.

-------------
Generated by KOReader %1.
]]),
    rev)

--[[-- Returns `true` if shown, `false` if the quickstart guide hasn't been
shown yet or if display is forced through a higher version number than when
it was first shown.
]]
function QuickStart:isShown()
    local shown_version = G_reader_settings:readSetting("quickstart_shown_version")
    return shown_version ~= nil and (shown_version >= self.quickstart_force_show_version)
end

--[[-- Generates the quickstart guide in the user's language and returns its location.

The fileformat is `quickstart-en-v2015.11-985-g88308992.html`, `en` being the
language of the generated file and `v2015.11-985-g88308992` the KOReader version
used to generate the file.

@treturn string path to generated HTML quickstart guide
]]
function QuickStart:getQuickStart()
    local quickstart_dir = ("%s/help"):format(DataStorage:getDataDir())
    if lfs.attributes(quickstart_dir, "mode") ~= "dir" then
        lfs.mkdir(quickstart_dir)
    end

    local quickstart_filename = ("%s/quickstart-%s-%s.html"):format(quickstart_dir, language, rev)
    if lfs.attributes(quickstart_filename, "mode") ~= "file" then
        -- purge old quickstart guides
        local iter, dir_obj = lfs.dir(quickstart_dir)
        for f in iter, dir_obj do
            if f:match("quickstart-.*%.html") then
                local file_abs_path = FFIUtil.realpath(("%s/%s"):format(quickstart_dir, f))
                os.remove(file_abs_path)
                DocSettings:open(file_abs_path):purge()
            end
        end

        local quickstart_html = FileConverter:mdToHtml(quickstart_guide, _("KOReader Quickstart Guide"))
        if quickstart_html then
            FileConverter:writeStringToFile(quickstart_html, quickstart_filename)
        end
    end
    -- remember filename for file manager
    self.quickstart_filename = quickstart_filename
    G_reader_settings:saveSetting("quickstart_shown_version", version)
    return quickstart_filename
end

return QuickStart
