package com.amazon.android.widget.items

import android.content.Context
import android.graphics.drawable.Drawable
import android.support.v4.content.ContextCompat
import com.amazon.android.widget.ICommandItemPresenter
import com.amazon.android.widget.ICommandItemPresenter.DisplayPreference
import com.amazon.kcp.application.AndroidApplicationController
import com.amazon.kcp.application.ErrorState
import com.amazon.kcp.font.DynamicFontDownloadHelper
import com.amazon.kcp.font.FontDownloadService
import com.amazon.kcp.font.FontUtils
import com.amazon.kcp.reader.AudibleHelper
import com.amazon.kcp.reader.ReaderActivity
import com.amazon.kcp.util.Utils
import com.amazon.kindle.build.BuildInfo
import com.amazon.kindle.krl.R
import com.amazon.kindle.krx.content.IBook

/**
 * Allows a user to start a download of a font if needed.
 * @author hensonm
 */
class DownloadFontsCommandItem(val activity: ReaderActivity)
    : AbstractReaderActionCommandItem(R.string.menuitem_download_fonts) {

    override fun getButtonIdentifier() = "DownloadFontsCommandItem"

    override fun getPriority(model: IBook?) = activity.resources.getInteger(R.integer.command_bar_download_fonts)

    override fun getDisplayPreference(context: Context): DisplayPreference = DisplayPreference.OVERFLOW_ONLY

    override fun onClick(): Boolean {
        val language = activity.docViewer?.bookInfo?.baseLanguage ?: return false

        val networkService = Utils.getFactory().networkService
        if (!networkService.hasNetworkConnectivity()) {
            AndroidApplicationController.getInstance().showAlert(ErrorState.CONNECTION_ERROR, null)
            return false
        }

        //Start font download
        val intent = FontDownloadService.getFontDownloadIntent(activity, language, false)
        this.activity.startService(intent)
        return true
    }

    override fun isVisible(model: IBook?): Boolean {
        val docViewer = activity.docViewer
        val isAudiblePlayerMode = AudibleHelper.isReaderInAudibleMode()

        if (BuildInfo.isEInkBuild() || isAudiblePlayerMode || docViewer?.bookInfo?.isFixedLayout != false) {
            return false
        }

        val language = docViewer.bookInfo.baseLanguage
        val appExpansionDynamicFont = DynamicFontDownloadHelper.DownloadableFonts.doesOnDemandFontExist(language)

        //if on demand font exists, we don't need to add the button, as it will download automagically
        if (appExpansionDynamicFont) {
            return false
        }

        //fall back to legacy font download check otherwise
        return !FontUtils.areRequiredFontsDownloaded(activity.appController.fileSystem, docViewer)
    }
}