// Copyright (C) 2013 Opera Software ASA. All rights reserved.
// This file is an original work developed by Opera Software ASA.

/**
 * This file is based on SavedPages.js file from iOS project.
 * It is used to save pages from WebView on Android 2.3.
 *
 */
'use strict';


var savedPages = {};

/**
 * Get the HTML of the document. Strips out all of the JavaScript
 * to avoid things like loading new external resources.
 *
 * @expose
 *
 * @return {string}
 */

savedPages.getHTML = function () {
  var elem, i, j;
  var root = document.documentElement.cloneNode(true);
  // Remove all JavaScript to prevent things like loading new resources
  // over the network
  var scripts = root.getElementsByTagName('script');
  for (i = scripts.length - 1; i >= 0; i--) {
    scripts[i].parentNode.removeChild(scripts[i]);
  }
  // Remove the most common inline events as well
  var evts = ['onchange', 'onclick', 'onfocus', 'onkeydown', 'onkeyup', 'onload',
              'onsubmit', 'onunload'];
  var elems = root.querySelectorAll('[' + evts.join('],[') + ']');
  // elems should be small on most pages
  i = elems.length;
  while (i--) {
    elem = elems[i];
    j = evts.length;
    while (j--) elem.removeAttribute(evts[j]);
  }
  var doctype = (document.compatMode === 'BackCompat') ? '' : '<!DOCTYPE html>';
  return doctype + root.outerHTML;
};



var ignoreTags = {
    "base":1,"BASE":1, "br":1,"BR":1, "data":1,"DATA":1, "datalist":1,"DATALIST":1,
    "head":1,"HEAD":1, "frame":1,"FRAME":1, "link":1,"LINK":1, "meta":1,"META":1,
    "param":1,"PARAM":1, "style":1,"STYLE":1, "noscript":1,"NOSCRIPT":1,
    "script":1,"SCRIPT":1, "source":1,"SOURCE":1};
var urlRe = /url\s*\(["']?([^data:'")]*)/;
var dataUriRe = /^data:/i;
/**
 * Get all references to external resources on the page.
 *
 * @expose
 *
 * @return {string} JSON
 */
savedPages.getResourceList = function () {
  var elems = document.querySelectorAll(
      "[src]:not(script), link[rel~='stylesheet'], link[rel~='icon']"
  );
  var resources = [];
  var i = elems.length, src;
  var matches;
  while (i--) {
    src = elems[i].src || elems[i].href;
    if (!dataUriRe.test(src)) resources[resources.length] = src;
  }

  elems = document.getElementsByTagName('*');
  i = elems.length;
  while (i--) {
    if (ignoreTags[elems[i].tagName]) continue;
    src = getComputedStyle(elems[i]).backgroundImage; // default "none"
    matches = src.match(urlRe);
    if (!matches) continue;
    if (!dataUriRe.test(matches[1])) {
      resources[resources.length] = matches[1];
    }
  }
  resources.toJSON = null;
  return JSON.stringify(resources);
};
