/**
 * WeRead Reader Mode Finder
 * 识别文章中的内容，寻找如标题、正文等有效信息，并对文章的结构和表现作出优化。
 * 基于 Safari Version 11.0 (13604.1.38.1.6)。
 *
 */

/**
 * JavaScript 原生语法扩展。
 */

String.prototype.lastInteger = function () {
    const e = /[0-9]+/g;
    var t = this.match(e);
    return t ? parseInt(t[t.length - 1]) : NaN;
};

/**
 *  polyfill window.getMatchedCSSRules() in Chrome
 *  @see https://www.chromestatus.com/features/4606972603138048 & https://gist.github.com/darrnshn/addeabe2575177342cc6242e20ecadbd
 */
if (typeof window.getMatchedCSSRules !== 'function') {
    var ELEMENT_RE = /[\w-]+/g,
        ID_RE = /#[\w-]+/g,
        CLASS_RE = /\.[\w-]+/g,
        ATTR_RE = /\[[^\]]+\]/g,
        // :not() pseudo-class does not add to specificity, but its content does as if it was outside it
        PSEUDO_CLASSES_RE = /\:(?!not)[\w-]+(\(.*\))?/g,
        PSEUDO_ELEMENTS_RE = /\:\:?(after|before|first-letter|first-line|selection)/g;

    // convert an array-like object to array
    function toArray(list) {
        return [].slice.call(list);
    }

    // handles extraction of `cssRules` as an `Array` from a stylesheet or something that behaves the same
    function getSheetRules(stylesheet) {
        var sheet_media = stylesheet.media && stylesheet.media.mediaText;
        // if this sheet is disabled skip it
        if (stylesheet.disabled) return [];
        // if this sheet's media is specified and doesn't match the viewport then skip it
        if (sheet_media && sheet_media.length && !window.matchMedia(sheet_media).matches) return [];
        // get the style rules of this sheet
        return toArray(stylesheet.cssRules);
    }

    function _find(string, re) {
        var matches = string.match(re);
        return re ? re.length : 0;
    }

    // calculates the specificity of a given `selector`
    function calculateScore(selector) {
        var score = [0, 0, 0],
            parts = selector.split(' '),
            part, match;
        //TODO: clean the ':not' part since the last ELEMENT_RE will pick it up
        while (part = parts.shift(), typeof part == 'string') {
            // find all pseudo-elements
            match = _find(part, PSEUDO_ELEMENTS_RE);
            score[2] = match;
            // and remove them
            match && (part = part.replace(PSEUDO_ELEMENTS_RE, ''));
            // find all pseudo-classes
            match = _find(part, PSEUDO_CLASSES_RE);
            score[1] = match;
            // and remove them
            match && (part = part.replace(PSEUDO_CLASSES_RE, ''));
            // find all attributes
            match = _find(part, ATTR_RE);
            score[1] += match;
            // and remove them
            match && (part = part.replace(ATTR_RE, ''));
            // find all IDs
            match = _find(part, ID_RE);
            score[0] = match;
            // and remove them
            match && (part = part.replace(ID_RE, ''));
            // find all classes
            match = _find(part, CLASS_RE);
            score[1] += match;
            // and remove them
            match && (part = part.replace(CLASS_RE, ''));
            // find all elements
            score[2] += _find(part, ELEMENT_RE);
        }
        return parseInt(score.join(''), 10);
    }

    // returns the heights possible specificity score an element can get from a give rule's selectorText
    function getSpecificityScore(element, selector_text) {
        var selectors = selector_text.split(','),
            selector, score, result = 0;
        while (selector = selectors.shift()) {
            if (element.webkitMatchesSelector(selector)) {
                score = calculateScore(selector);
                result = score > result ? score : result;
            }
        }
        return result;
    }

    function sortBySpecificity(element, rules) {
        // comparing function that sorts CSSStyleRules according to specificity of their `selectorText`
        function compareSpecificity(a, b) {
            return getSpecificityScore(element, b.selectorText) - getSpecificityScore(element, a.selectorText);
        }

        return rules.sort(compareSpecificity);
    }

    //TODO: not supporting 2nd argument for selecting pseudo elements
    //TODO: not supporting 3rd argument for checking author style sheets only
    window.getMatchedCSSRules = function (element /*, pseudo, author_only*/) {
        var style_sheets, sheet, sheet_media,
            rules, rule,
            result = [];
        // get stylesheets and convert to a regular Array
        style_sheets = toArray(window.document.styleSheets);

        // assuming the browser hands us stylesheets in order of appearance
        // we iterate them from the beginning to follow proper cascade order
        while (sheet = style_sheets.shift()) {
            // get the style rules of this sheet
            rules = getSheetRules(sheet);
            // loop the rules in order of appearance
            while (rule = rules.shift()) {
                // if this is an @import rule
                if (rule.styleSheet) {
                    // insert the imported stylesheet's rules at the beginning of this stylesheet's rules
                    rules = getSheetRules(rule.styleSheet).concat(rules);
                    // and skip this rule
                    continue;
                }
                // if there's no stylesheet attribute BUT there IS a media attribute it's a media rule
                else if (rule.media) {
                    // insert the contained rules of this media rule to the beginning of this stylesheet's rules
                    rules = getSheetRules(rule).concat(rules);
                    // and skip it
                    continue
                }
                //TODO: for now only polyfilling Gecko
                // check if this element matches this rule's selector
                if (element.webkitMatchesSelector(rule.selectorText)) {
                    // push the rule to the results set
                    result.push(rule);
                }
            }
        }
        // sort according to specificity
        return sortBySpecificity(element, result);
    };
}

/**
 * Reader Finder 业务。
 */

/**
 * 搜寻文章的评分体系相关数值。
 */
const ReaderMinimumScore = 1600,
    ReaderMinimumAdvantage = 15,
    ArticleMinimumScoreDensity = 4.25;

/**
 * 候选文章相关判断条件。
 */
const CandidateMinimumWidth = 280,
    CandidateMinimumHeight = 295,
    CandidateMinimumArea = 170000,
    CandidateMaximumTop = 1300,
    CandidateMinimumWidthPortionForIndicatorElements = .5,
    CandidateMinumumListItemLineCount = 4,
    CandidateTagNamesToIgnore = {
        A: 1,
        EMBED: 1,
        FORM: 1,
        HTML: 1,
        IFRAME: 1,
        OBJECT: 1,
        OL: 1,
        OPTION: 1,
        SCRIPT: 1,
        STYLE: 1,
        svg: 1,
        UL: 1
    },
    PrependedArticleCandidateMinimumHeight = 50,
    AppendedArticleCandidateMinimumHeight = 200,
    AppendedArticleCandidateMaximumVerticalDistanceFromArticle = 150,
    StylisticClassNames = { // 装饰性的 className
        justfy: 1,
        justify: 1,
        left: 1,
        right: 1,
        small: 1
    };

/**
 * 文章中各种内容的属性和标签名的判断条件。
 */
const CommentRegEx = /comment|meta|footer|footnote/,
    CommentMatchPenalty = 0.75,
    ArticleRegEx = /(?:(?:^|\s)(?:(post|hentry|entry)[-_]?(?:content|text|body)?|article[-_]?(?:content|text|body|page)?)(?:\s|$))/i,
    ArticleMatchBonus = 0.5,
    CarouselRegEx = /carousel/i,
    CarouselMatchPenalty = 0.75,
    TitleCandidateDepthScoreMultiplier = 0.1,
    StringSimilarityToDeclareStringsNearlyIdentical = 0.97, // 如果两个字符串的相似度达到该值，则可以认为基本相同。
    StringSimilarityToDeclareStringsIsTheEndOfTheArticle = 0.6,
    SectionRegex = /section|content.*component/i,
    DensityExcludedElementSelector = "#disqus_thread, #comments, .userComments",
    PositiveRegEx = /article|body|content|entry|hentry|page|pagination|post|related-asset|text/i, // 很可能是文章
    NegativeRegEx = /advertisement|breadcrumb|combx|comment|contact|disqus|footer|link|meta|mod-conversations|promo|related|scroll|share|shoutbox|sidebar|social|sponsor|subscribe|tags|toolbox|widget|[-_]ad$|zoom-(in|out)/i, // 很可能不是文章
    VeryPositiveClassNameRegEx = /instapaper_body/, // 一定要保留
    VeryNegativeClassNameRegEx = /instapaper_ignore|weapp_display_element|weapp_card/, // 一定要过滤掉
    SharingRegex = /email|print|rss|digg|slashdot|delicious|reddit|share|twitter|facebook|pinterest|whatsapp/i,
    QRCodeTitleRegEx = /qrcode/,
    QRCodeInstructionRegEx = /((识别|扫描|长按|扫)(）|\))?)(图中)?二维码|阅读原文|长按(上|下)(方|面|图)|扫描(上|下)(方|面|图)|后台回复|分享到朋友圈|转载必须获得授权|点击进入\S{2}号|长按识别|点开图片|赞赏|扫码关注|微信公众号|微信号/,
    KeywordsAboutTheEndOfTheArticle = /The End|End|往期回顾|推荐阅读(点字即看)?|精彩回顾|热文推荐|本期完|猜您喜欢|如果漏过下列文章|本期完|长按指纹|长按下方扫码关注|新朋友别忘了关注哦|延伸阅读|近期文章精选|往期精彩|赞赏二维码|每日一题|热门考点回顾|大家都在读|更多精彩内容|相关(文章)?推荐|本文版权归原作者及公众号所有，图片均来源于网络|手机应用商店搜索.*点击下载|(（)?未完待续(）)?/ig, // 文章结尾关键词，关键词往后的内容应该去掉
    VeryLiberalCommentRegex = /comment/i,
    AdvertisementHostRegex = /^adserver\.|doubleclick.net$/i,
    SidebarRegex = /sidebar/i,
    LazyLoadRegex = /lazy/i,
    AppleDotComAndSubdomainsRegex = /.*\.apple\.com\.?$/,
    MPDomainRegex = /^mp\.weixin\.qq\.com\.?$/,
    SchemaDotOrgArticleContainerSelector = "*[itemtype='https://schema.org/Article'], *[itemtype='https://schema.org/NewsArticle'], *[itemtype='http://schema.org/Article'], *[itemtype='http://schema.org/NewsArticle']";

/**
 * 文章中各种内容的尺寸比例判断条件。
 */
const HeaderMinimumDistanceFromArticleTop = 200,
    HeaderLevenshteinDistanceToLengthRatio = 0.75,
    MinimumRatioOfListItemsBeingRelatedToSharingToPruneEntireList = 0.5,
    FloatMinimumHeight = 130,
    ImageSizeTiny = 32,
    ToleranceForLeadingImageWidthToArticleWidthForFullWidthPresentation = 50,
    MaximumFloatWidth = 325,
    AnchorImageMinimumWidth = 100,
    AnchorImageMinimumHeight = 100,
    MinimumHeightForImagesAboveTheArticleTitle = 50,
    MainImageMinimumWidthAndHeight = 83,
    BaseFontSize = 16,
    BaseLineHeightRatio = 1.125,
    MaximumExactIntegralValue = 9007199254740992,
    TextNodeLengthPower = 1.25,
    FindArticleMode = {
        Element: false,
        ExistenceOfElement: true
    },
    CleaningType = {
        MainArticleContent: 0,
        MetadataContent: 1,
        LeadingFigure: 2
    },
    MaximumWidthOrHeightOfImageInMetadataSection = 20;

/**
 * 屏幕尺寸比例相关的判断条件。
 */
const MinimumAverageDistanceBetweenHRElements = 400,
    MinimumAverageDistanceBetweenHeaderElements = 400,
    PortionOfCandidateHeightToIgnoreForHeaderCheck = .1,
    DefaultNumberOfTextNodesToCheckForLanguageMultiplier = 3,
    NumberOfCharactersPerTextNodeToEvaluateForLanguageMultiplier = 12,
    MinimumRatioOfCharactersForLanguageMultiplier = .5,
    ScoreMultiplierForChineseJapaneseKorean = 3,
    MinimumContentMediaHeight = 150,
    MinimumContentMediaWidthToArticleWidthRatio = .25,
    MaximumContentMediaAreaToArticleAreaRatio = .2;

/**
 * 超链接相关的判断条件。
 */
const LinkContinueMatchRegEx = /continue/gi,
    LinkNextMatchRegEx = /next/gi,
    LinkPageMatchRegEx = /page/gi,
    LinkListItemBonus = 5,
    LinkPageMatchBonus = 10,
    LinkNextMatchBonus = 15,
    LinkContinueMatchBonus = 15,
    LinkNextOrdinalValueBase = 3,
    LinkMismatchValueBase = 2,
    LinkMatchWeight = 200,
    LinkMaxVerticalDistanceFromArticle = 200,
    LinkVerticalDistanceFromArticleWeight = 150,
    LinkCandidateXPathQuery = "descendant-or-self::*[(not(@id) or (@id!='disqus_thread' and @id!='comments')) and (not(@class) or @class!='userComments')]/a",
    LinkDateRegex = /\D(?:\d\d(?:\d\d)?[\-\/](?:10|11|12|0?[1-9])[\-\/](?:30|31|[12][0-9]|0?[1-9])|\d\d(?:\d\d)?\/(?:10|11|12|0[1-9])|(?:10|11|12|0?[1-9])\-(?:30|31|[12][0-9]|0?[1-9])\-\d\d(?:\d\d)?|(?:30|31|[12][0-9]|0?[1-9])\-(?:10|11|12|0?[1-9])\-\d\d(?:\d\d)?)\D/,
    LinkURLSearchParameterKeyMatchRegex = /(page|^p$|^pg$)/i,
    LinkURLPageSlashNumberMatchRegex = /\/.*page.*\/\d+/i,
    LinkURLSlashDigitEndMatchRegex = /\/\d+\/?$/,
    LinkURLArchiveSlashDigitEndMatchRegex = /archives?\/\d+\/?$/,
    LinkURLBadSearchParameterKeyMatchRegex = /author|comment|feed|id|nonce|related/i,
    LinkURLSemanticMatchBonus = 100,
    LinkMinimumURLSimilarityRatio = .75;

/**
 * Twitter 相关内容的判断。
 */
const TweetURLRegex = /^https?:\/\/(.+\.)?twitter\.com\/.*\/status\/(.*\/)*[0-9]+\/?$/i,
    TweetIframeTitleRegex = /tweet/i;

/**
 * 通过 hostname 判断是否为内嵌视频资源的网站
 *
 * @param hostname
 * @returns {boolean}
 */
function hostnameMatchesHostKnownToContainEmbeddableMedia(hostname) {
    const t = /^(.+\.)?(v\.qq\.com|imgcache\.qq\.com|ugcbsy\.qq\.com|youtube\.com|vimeo\.com|dailymotion\.com|soundcloud\.com|mixcloud\.com|embedly\.com|embed\.ly)\.?$/;
    return t.test(hostname);
}

/**
 * 获取懒加载图片的 url，因为页面上的懒加载图片的 src 值通常都不是实际的原图 url，因此需要通过分析获取真正的图片 url。
 * @param currentCloneElement 需要被分析的 img 的复制 element。
 * @param currentElementClassName 需要被分析的 img 标签的 class-name。
 * @returns 返回懒加载图片的 url。
 */
function lazyLoadingImageURLForElement(currentCloneElement, currentElementClassName) {
    // 1px * 1px 大小的透明图，号称"体积最小占位图"，因而被大量网站使用，如果 img 的 src 为这张图片的 base64 值，则肯定使用了懒加载图片。
    const placeholderImage = /(data:image\/)?gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAAABAAEAAAICTAEAOw==/,
    mpPlaceHolderImage = /(data:image\/)?gif;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVQImWNgYGBgAAAABQABh6FO1AAAAABJRU5ErkJggg==/,
        srcAttributes = {
            "data-lazy-src": 1,
            "data-original": 1,
            datasrc: 1,
            "data-src": 1,
            "original-src": 1,
            "rel:bf_image_src": 1,
            "deferred-src": 1,
            "data-mediaviewer-src": 1
        }, // 优先尝试从以上属性中获取图片真正的 url
        possibleAttributes = {
            original: 1
        }; // 次要的可能包含图片真正 url 的属性

    // 获取 img 标签上的 src 属性值
    var currentSrcValue = currentCloneElement.getAttribute("src");

    // 判断 img 标签的 src 属性是否为上面的"体积最小占位图"或者是其他包含"transparent"和"empty"字样的图片。
    var isPlaceholderImage = /transparent|empty/i.test(currentSrcValue) || placeholderImage.test(currentSrcValue) || mpPlaceHolderImage.test(currentSrcValue);

    // 遍历两次，也就是遍历 img 标签以及其父元素中是否包含 srcAttributes 或 possibleAttributes 中列举的属性。
    const traversalCount = 2;
    var tempElement = currentCloneElement;
    for (var elementIndex = 0; tempElement && elementIndex < traversalCount; tempElement = tempElement.parentElement, ++elementIndex) {
        // 遍历 img 标签以及其父元素

        for (var elementAttributes = tempElement.attributes, attributeIndex = 0; attributeIndex < elementAttributes.length; ++attributeIndex) {
            // 遍历属性

            var currentAttribute = elementAttributes[attributeIndex].nodeName;
            if (srcAttributes[currentAttribute.toLowerCase()]) {
                // 如果当前属性名为 srcAttributes 中有列举的属性名，则以当前属性的值为 url 结果。
                return tempElement.getAttribute(currentAttribute);
            }

            var isImageRelated = /\.(jpe?g|png|gif|bmp)$/i.test(elementAttributes[currentAttribute].value);
            if (possibleAttributes[currentAttribute.toLowerCase()] && isImageRelated) {
                // 如果当前属性名为 possibleAttributes 中的属性名，并且属性值包含图片扩展名，则以当前属性的值为 url 结果。
                return tempElement.getAttribute(currentAttribute);
            }

            if (isPlaceholderImage && /^data.*(src|source)$/i.test(currentAttribute) && isImageRelated) {
                // 如果 img 标签的 src 属性是否为上面的"体积最小占位图"或者是其他包含"transparent"和"empty"字样的图片，并且当前属性名与 data-src 和 data-source 相关，同时属性值包含图片扩展名，则以当前属性的值为 url 结果。
                return tempElement.getAttribute(currentAttribute);
            }
        }
    }
    if (LazyLoadRegex.test(currentElementClassName) && "function" === typeof URL) {
        // 如果上面的逻辑无法命中图片真正的 url，则进入本逻辑。
        // 需要被分析的 img 标签的 class-name 包含"lazy"等字样，并且 URL 对象可用，则能进行这一步。
        var currentElementSrc;
        try {
            // 使用 img 标签的 src 构建一个 URL 对象。
            currentElementSrc = new URL(currentCloneElement.src)
        } catch (p) {
        }
        if (currentElementSrc && currentElementSrc.search) {
            // 如果 img 标签的 src 中包含查询字符（即 url 中的 ?xxx 这部分内容），则获取查询字符中是否包含宽高信息（以"w"，"width"，"h"，"height"四个关键字判断）
            var widthParameterOfUrl,
                heightParameterOfUrl;
            const widthKeywords = ["w", "width"];
            for (var i = 0; i < widthKeywords.length; ++i) {
                var keywork = widthKeywords[i],
                    widthValueOfUrl = currentElementSrc.searchParams.get(keywork);
                if (widthValueOfUrl && !isNaN(parseInt(widthValueOfUrl))) {
                    // 得到 url 中的宽度参数名
                    widthParameterOfUrl = keywork;
                    break;
                }
            }
            const heightKeywords = ["h", "height"];
            for (var j = 0; j < heightKeywords.length; ++j) {
                var keyword = heightKeywords[j],
                    heightValueOfUrl = currentElementSrc.searchParams.get(keyword);
                if (heightValueOfUrl && !isNaN(parseInt(heightValueOfUrl))) {
                    // 得到 url 中的高度参数名
                    heightParameterOfUrl = keyword;
                    break;
                }
            }
            if (widthParameterOfUrl && heightParameterOfUrl) {
                // 如果查询字符中包含宽高信息，则获取 img 标签上的 width 和 height 属性。
                var x = currentCloneElement.getAttribute("width"),
                    T = currentCloneElement.getAttribute("height");
                if (!isNaN(parseInt(x)) && !isNaN(parseInt(T))) {
                    // 如果 img 标签上确实能获取 width 和 height 属性，则把查询字符中获取的宽高信息和 img 标签属性上的宽高信息写入 url，并返回该 url。
                    // 通常这类 url 中获取的结果是尺寸较小的图片版本（即图片 url 能获取到需要的图片，但查询字符中的参数控制了获取的图片为较小尺寸的图片），
                    // 因此把 img 标签的 width 和 height 属性更新到 url，后续该 url 就可以获取到实际需要的尺寸的图片。
                    return currentElementSrc.searchParams.set(widthParameterOfUrl, x), currentElementSrc.searchParams.set(heightParameterOfUrl, T), currentElementSrc.href
                }
            }
        }
    }
    return null;
}

/**
 * 去除 HTML 标签中多余的属性。
 */
function sanitizeElementByRemovingAttributes(element) {
    const excessProperty = /^on|^id$|^class$|^style$|^autofocus$/;
    var elementAttributes = element.attributes;
    for (var i = 0; i < elementAttributes.length; ++i) {
        var attributeName = elementAttributes[i].nodeName;
        if (excessProperty.test(attributeName)) {
            element.removeAttribute(attributeName);
            i--;
        }
    }
}

/**
 * 判断是否为简体/繁体中文、日语、韩语和阿拉伯语这几种亚洲文字，这些文字在计算文字密度（文字占屏幕大小的比例）时会乘以 12，作为提权，
 * 因为同样的字符数，对于亚洲文字通常能表达更多的含义，因此亚洲文字需要乘以更高的权重。
 * @param character 需要被检测的字符。
 */
function characterNeedsScoreMultiplier(character) {
    if (!character || character.length === 0) {
        return false;
    }
    // 获取字符串的第一个字符，转换为 UTF-16，结果为对应的 10 进制数字
    // 把以下数字转换为对应的文字的方法：
    // 1. 10进制转换为16进制：http://tool.oschina.net/hexconvert，得到的应该是4或5位的16进制数字
    // 2. 16进制数字转换为 UTF-16，进入 http://www.qqxiuzi.cn/bianma/Unicode-UTF.php，选择 UTF-16 编码，
    // 4 位数前面加上 FEFF 即可转换，5位数前面则加上 FEF 进行转换。
    // 例如 19968 转为16进制是 4e00，因此把 FEFF4e00 进行转换，得到的 UTF 字符是汉字"一"。
    var characterCode = character.charCodeAt(0);
    if (characterCode > 11904 && characterCode < 12031) {
        // 应该是汉字部首，16进制表示：\u2e80 至 \u9fa5
        return true;
    }
    if (characterCode > 12352 && characterCode < 12543) {
        // 日语，16进制表示：\u3040 至 \u30ff
        return true;
    }
    if (characterCode > 12736 && characterCode < 19903) {
        // 应该是汉字笔画，16进制表示：\u31c0 至 \u4dbf
        return true;
    }
    if (characterCode > 19968 && characterCode < 40959) {
        // 简体中文，16进制表示：\u4e00（对应汉字是"一"）至 \u9fa5（对应汉字是"龥"）
        return true;
    }
    if (characterCode > 44032 && characterCode < 55215) {
        // 韩语，16进制表示：\uac00 至 \ud7af
        return true;
    }
    if (characterCode > 63744 && characterCode < 64255) {
        // 简体中文，16进制表示：\u（对应汉字是"豈"）至 \u9fa5（对应汉字是个奇奇怪怪的字）
        return true;
    }
    if (characterCode > 65072 && characterCode < 65103) {
        // 各种符号，16进制表示：\ufe30 至 \ufe4f
        return true;
    }
    if (characterCode > 131072 && characterCode < 173791) {
        // 阿拉伯语
        return true;
    }
    if (characterCode > 194560 && characterCode < 195103) {
        // 阿拉伯相关的符号
        return true;
    }
    return false;
}

/**
 * 计算两个 DOM 的距离
 * @param elementA 参与计算的 DOM.
 * @param elementB 参与计算的 DOM。
 * @param distanceCap 距离的上限，如果提供了上限，则计算值不能超过该上限值。
 * @returns 返回两个 DOM 相差的层级数。
 */
function domDistance(elementA, elementB, distanceCap) {
    var ancestorsA = [];
    var node = elementA;
    while (node) {
        ancestorsA.unshift(node);
        node = node.parentNode;
    }
    var ancestorsB = [];
    node = elementB;
    while (node) {
        ancestorsB.unshift(node);
        node = node.parentNode;
    }
    var shallowerDepth = Math.min(ancestorsA.length, ancestorsB.length);
    var domDistance = Math.abs(ancestorsA.length - ancestorsB.length);
    for (var i = shallowerDepth; i >= 0; i--) {
        if (ancestorsA[i] === ancestorsB[i]) {
            break;
        }
        domDistance += 2;
        if (distanceCap && domDistance >= distanceCap) {
            return distanceCap;
        }
    }
    return domDistance;
}

function fontSizeFromComputedStyle(style, fallbackSize) {
    var fontSize = parseInt(style.fontSize);
    if (isNaN(fontSize)) {
        fontSize = fallbackSize ? fallbackSize : BaseFontSize;
    }
    return fontSize;
}

/**
 * 传入节点，计算主体内容的样式，在计算各种元素时会使用到这个主体内容的样式作对比。
 * @param contentDocument document
 * @param node 对传入的节点计算其中的主体样式。
 * @returns 返回计算出的主体样式。
 */
function contentTextStyleForNode(contentDocument, node) {
    function getComputedStyleIfNodeIsBodyText(textNode) {
        if (isNodeWhitespace(textNode)) {
            return null;
        }
        var style = getComputedStyle(textNode.parentNode);
        if (style['float'] !== "none") {
            return null;
        }
        return style;
    }

    var nonHeaderTextXPathQuery = "descendant::text()[not(parent::h1) and not(parent::h2) and not(parent::h3) and not(parent::h4) and not(parent::h5) and not(parent::h6)]"; // 使用 XPath 查询，排除非标题元素
    var articleTextNodes = contentDocument.evaluate(nonHeaderTextXPathQuery, node, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null); // 获取所有符合查询条件的 DOM
    for (var nodeIndex = 0; nodeIndex < articleTextNodes.snapshotLength; ++nodeIndex) {
        var nodeItem = articleTextNodes.snapshotItem(nodeIndex),
            notArticleText = false;
        for (var nodeItemParent = nodeItem.parentElement; nodeItemParent !== node; nodeItemParent = nodeItemParent.parentElement) {
            // 检测 DOM 的 class-name 是否包含"很可能不是文章"的关键词，如果包含则排除它。
            if (NegativeRegEx.test(nodeItemParent.className)) {
                notArticleText = true;
                break;
            }
        }
        if (!notArticleText) {
            var style = getComputedStyleIfNodeIsBodyText(nodeItem);
            if (style) {
                return style;
            }
        }
    }
    return null;
}

function isNodeWhitespace(node) {
    return node && node.nodeType === Node.TEXT_NODE ? !/\S/.test(node.data) : false;
}

function removeWhitespace(string) {
    return string.replace(/\s+/g, '');
}

function isElementNode(node) {
    return node && node.nodeType === Node.ELEMENT_NODE;
}

/**
 * 通过元素的 style 判断元素是否有因为设置了 clip 而导致不可见（clip 的 left > right 或者 top > bottom 时则整个元素不可见）。
 * @param style 元素的 style。
 * @returns {boolean} 元素是否因为 clip 而不可见。
 */
function computedStyleIndicatesElementIsInvisibleDueToClipping(style) {
    if (style.position !== 'absolute') {
        // CSS Clip 仅对 position: absolute 元素生效。
        return false;
    }
    // style.clip 的结果在 Chrome 下为 rect(10px 10px 10px 10px)
    // Safari 下为 rect(10px, 10px, 10px, 10px)
    // 因此正则匹配需要兼容两种情况。
    // 匹配到的结果示例：["rect(10px 10px 10px 10px)", "10px", "10px", "10px", "10px"]
    var clipValue = style.clip.match(/^rect\((\d+px|auto),? (\d+px|auto),? (\d+px|auto),? (\d+px|auto)\)$/);
    if (!clipValue || clipValue.length !== 5) {
        return false;
    }
    var directionValues = clipValue.map(function (value) {
            return parseInt(value);
        }),
        top = directionValues[1];
    if (isNaN(top)) {
        top = 0;
    }
    var right = directionValues[2],
        bottom = directionValues[3],
        left = directionValues[4];
    if (isNaN(left)) {
        left = 0;
    }
    return top >= bottom || left >= right;
}

function isElementVisible(element) {
    var style = getComputedStyle(element);
    if (style.visibility !== 'visible' || style.display === 'none') {
        return false;
    }
    if (cachedElementBoundingRect(element).height) {
        return true;
    }
    var range = document.createRange();
    range.selectNode(element);
    return !!range.getBoundingClientRect().height;
}

function isElementPositionedOffScreen(element) {
    var boundingRect = cachedElementBoundingRect(element);
    if (boundingRect.height && boundingRect.width) {
        return boundingRect.bottom <= 0 || boundingRect.right <= 0;
    } else {
        return false;
    }
}

function elementDepth(element) {
    var depth = 0;
    while (element) {
        element = element.parentElement;
        depth++;
    }
    return depth;
}

/**
 * 计算一个元素属于某个元素的第几层子元素。
 * @param element
 * @param ancestorElement
 * @returns {*}
 */
function depthOfElementWithinElement(element, ancestorElement) {
    for (var i = 0; element !== ancestorElement; element = element.parentElement) {
        if (!element) {
            return NaN;
        }
        i++;
    }
    return i;
}

/**
 * 根据标签名寻找最相近的祖先元素。
 * @param element
 * @param targetTagName
 * @param discontinuity 中断条件，如果寻找父级时找到条件中的元素，则中断寻找，直接返回该元素。
 * @returns 返回最相近的祖先元素。
 */
function nearestAncestorElementWithTagName(element, targetTagName, discontinuity) {
    var discontinuityElement = {};
    if (discontinuity) {
        for (var i = 0; i < discontinuity.length; ++i) {
            discontinuityElement[discontinuity[i]] = true;
        }
    }
    if (discontinuityElement[element.tagName]) {
        return null;
    }

    while (element = element.parentElement) {
        var tagName = element.tagName;
        if (discontinuityElement[tagName]) {
            break;
        }
        if (tagName === targetTagName) {
            return element;
        }
    }
    return null;
}

function cachedElementBoundingRect(element) {
    if (element._cachedElementBoundingRect) {
        return element._cachedElementBoundingRect;
    }

    var rect = element.getBoundingClientRect();
    MPReaderModeController._elementsWithCachedBoundingRects.push(element);
    if (MPReaderModeController._cachedScrollX || MPReaderModeController._cachedScrollY) {
        element._cachedElementBoundingRect = {
            top: rect.top + MPReaderModeController._cachedScrollY,
            right: rect.right + MPReaderModeController._cachedScrollX,
            bottom: rect.bottom + MPReaderModeController._cachedScrollY,
            left: rect.left + MPReaderModeController._cachedScrollX,
            width: rect.width,
            height: rect.height
        };
    } else {
        element._cachedElementBoundingRect = rect;
    }
    return element._cachedElementBoundingRect;
}

function clearCachedElementBoundingRects() {
    var elementsWithRects = MPReaderModeController._elementsWithCachedBoundingRects;
    for (var n = 0; n < elementsWithRects.length; ++n) {
        elementsWithRects[n]._cachedElementBoundingRect = null;
    }
    MPReaderModeController._elementsWithCachedBoundingRects = [];
}

function trimmedInnerTextIgnoringTextTransform(element) {
    var innerText = element.innerText;
    if (!/\S/.test(innerText)) {
        return element.textContent.trim();
    }
    var style = getComputedStyle(element),
        textTransformStyle = style.textTransform;

    if (textTransformStyle === 'uppercase' || textTransformStyle === 'lowercase') {
        return element.textContent.trim();
    } else {
        return innerText.trim();
    }
}

/**
 * 判断两个字符串的相似程度，采用莱文斯坦距离算法（由一个转成另一个所需的最少编辑操作次数）。
 * @param string1 需要进行比较的字符串1。
 * @param string2 需要进行比较的字符串2。
 * @returns 返回莱文斯坦距离值。
 *
 * @see https://en.wikipedia.org/wiki/Levenshtein_distance
 */
function levenshteinDistance(string1, string2) {
    var lengthOfString1 = string1.length;
    var lengthOfString2 = string2.length;
    var d = new Array(lengthOfString1 + 1);
    for (var i = 0; i < lengthOfString1 + 1; i++) {
        d[i] = new Array(lengthOfString2 + 1);
        d[i][0] = i;
    }
    for (var j = 0; j < lengthOfString2 + 1; j++)
        d[0][j] = j;
    for (var j = 1; j < lengthOfString2 + 1; j++) {
        for (var i = 1; i < lengthOfString1 + 1; i++) {
            if (string1[i - 1] === string2[j - 1])
                d[i][j] = d[i - 1][j - 1];
            else {
                var deletion = d[i - 1][j] + 1;
                var insertion = d[i][j - 1] + 1;
                var substitution = d[i - 1][j - 1] + 1;
                d[i][j] = Math.min(deletion, insertion, substitution);
            }
        }
    }
    return d[lengthOfString1][lengthOfString2];
}

/**
 * 计算两个字符串的相似度，相似度 = (两个字符中较大的那个字符串的长度 - 一个转成另一个所需的最少编辑操作次数) / 两个字符中较大的那个字符串的长度。
 * @param string1 需要进行比较的字符串1。
 * @param string2 需要进行比较的字符串2。
 *
 * @see #levenshteinDistance
 */
function stringSimilarity(string1, string2) {
    var maxLength = Math.max(string1.length, string2.length);
    return maxLength ? (maxLength - levenshteinDistance(string1, string2)) / maxLength : 0;
}

/**
 * 判断两个字符串是否接近相同（通过判断相似度是否很接近）。
 * @param string1 需要进行比较的字符串1。
 * @param string2 需要进行比较的字符串2。
 *
 * @see #stringSimilarity
 */
function stringsAreNearlyIdentical(string1, string2) {
    return string1 === string2 ? true : stringSimilarity(string1, string2) > StringSimilarityToDeclareStringsNearlyIdentical;
}

function elementIsCommentBlock(element) {
    if (/(^|\s)comment/.test(element.className)) {
        return true;
    }
    var elementId = element.getAttribute('id');
    return elementId && elementId.indexOf('comment') === 0;
}

/**
 * 判断元素是否为 Twitter iFrame，以便后续做特殊处理。
 * @param element
 * @returns {boolean}
 */
function elementLooksLikeEmbeddedTweet(element) {
    if (element.tagName !== 'IFRAME') {
        return false;
    }
    if (!element.contentDocument) {
        return false;
    }
    var documentElement = element.contentDocument.documentElement,
        relatedElementCount = 0, // 记录 twitter 相关的子元素的个数，若找到2个或以上相关的子元素，则认为 element 是 Twitter iFrame
        blockquoteElement = documentElement.querySelector('blockquote');

    if (blockquoteElement && TweetURLRegex.test(blockquoteElement.getAttribute("cite"))) {
        relatedElementCount++;
    }
    var iframeElement = documentElement.querySelector("[data-iframe-title]");
    if (iframeElement && TweetIframeTitleRegex.test(iframeElement.getAttribute("data-iframe-title"))) {
        relatedElementCount++;
    }
    if (element.classList.contains("twitter-tweet")) {
        relatedElementCount++;
    }
    if (documentElement.querySelector("[data-tweet-id]")) {
        relatedElementCount++;
    }
    return relatedElementCount > 2;
}

/**
 * 判断是否是图片轮播组件。
 * @param element
 */
function elementLooksLikePartOfACarousel(element) {
    const t = /carousel-|carousel_|-carousel|_carousel/,
        depth = 3;
    for (var currentElement = element, i = 0; depth > i; i++) {
        if (!currentElement) {
            return false;
        }
        if (t.test(currentElement.className) || t.test(currentElement.getAttribute('data-analytics'))) {
            return true;
        }
        currentElement = currentElement.parentElement;
    }
}

/**
 * 判断一个 iframe 是否需要去除。
 * @param element iframe 元素。
 * @param contentDocument 一个 document 实例。
 */
function shouldPruneIframe(element, contentDocument) {
    if (element.srcdoc) {
        return true;
    } else {
        if (hostnameMatchesHostKnownToContainEmbeddableMedia(anchorForURL(element.src, contentDocument).hostname)) {
            return false;
        } else {
            return !elementLooksLikeEmbeddedTweet(element.originalElement);
        }
    }
}

/**
 * 计算文字节点的分值权重，后续用于判断文字的语义。
 * 具体来说，如果文字节点中的亚洲语言超过一定比例（MinimumRatioOfCharactersForLanguageMultiplier），
 * 则认为传入的文字节点为亚洲语言的文字，权重会更高。
 *
 * @see #characterNeedsScoreMultiplier
 */
function languageScoreMultiplierForTextNodes(textNodes) {
    if (!textNodes || !textNodes.length) {
        return 1;
    }
    var effectiveTextNodesCount = Math.min(textNodes.length, DefaultNumberOfTextNodesToCheckForLanguageMultiplier),
        characterNeedsScoreMultiplierCount = 0,
        effectiveStringCountInTextNodes = 0;
    for (var i = 0; i < effectiveTextNodesCount; i++) {
        var string = textNodes[i].nodeValue.trim(),
            effectiveStringCount = Math.min(string.length, NumberOfCharactersPerTextNodeToEvaluateForLanguageMultiplier);
        for (var j = 0; j < effectiveStringCount; j++) {
            if (characterNeedsScoreMultiplier(string[j])) {
                // 记录中文、日文和韩文等亚洲语言的字符数
                characterNeedsScoreMultiplierCount++;
            }
        }
        effectiveStringCountInTextNodes += effectiveStringCount;
    }
    if (characterNeedsScoreMultiplierCount >= effectiveStringCountInTextNodes * MinimumRatioOfCharactersForLanguageMultiplier) {
        return ScoreMultiplierForChineseJapaneseKorean;
    } else {
        return 1;
    }
}

/**
 * 根据元素的标签名、ID、class-name 计算分值权重。
 */
function scoreMultiplierForElementTagNameAndAttributes(element) {
    var scoreMultiplier = 1;
    for (var currentElement = element; currentElement; currentElement = currentElement.parentElement) {
        var elementID = currentElement.getAttribute('id');
        if (elementID && ArticleRegEx.test(elementID)) {
            scoreMultiplier += ArticleMatchBonus;
        }
        if (CommentRegEx.test(elementID)) {
            scoreMultiplier -= CommentMatchPenalty;
        }
        if (CarouselRegEx.test(elementID)) {
            scoreMultiplier -= CarouselMatchPenalty;
        }

        var className = currentElement.className;
        if (className && ArticleRegEx.test(className)) {
            scoreMultiplier += ArticleMatchBonus;
        }
        if (CommentRegEx.test(className)) {
            scoreMultiplier -= CommentMatchPenalty;
        }
        if (CarouselRegEx.test(className)) {
            scoreMultiplier -= CarouselMatchPenalty;
        }
        if (currentElement.tagName === 'ARTICLE') {
            scoreMultiplier += ArticleMatchBonus;
        }
    }
    return scoreMultiplier < 0 ? 0 : scoreMultiplier;
}

// TODO:Kayo 涉及 ReaderArticleFinderJSController，后续处理
function elementAtPoint(e, t) {
    if ("undefined" != typeof ReaderArticleFinderJSController && ReaderArticleFinderJSController.nodeAtPoint) {
        var n = ReaderArticleFinderJSController.nodeAtPoint(e, t);
        return n && n.nodeType !== Node.ELEMENT_NODE && (n = n.parentElement), n
    }
    return document.elementFromPoint(e, t)
}

// TODO:Kayo 涉及 ReaderArticleFinderJSController，后续处理
function userVisibleURLString(e) {
    return "undefined" != typeof ReaderArticleFinderJSController && ReaderArticleFinderJSController.userVisibleURLString ? ReaderArticleFinderJSController.userVisibleURLString(e) : e
}

function anchorRunsJavaScriptOnActivation(element) {
    var href = element.href;
    return href.trim().substring(0, 11).toLowerCase() === 'javascript:';
}

function anchorForURL(url, contentDocument) {
    var anchorElement = contentDocument.createElement('a');
    anchorElement.href = url;
    return anchorElement;
}

function anchorLinksToAttachment(element) {
    return /\battachment\b/i.test(element.getAttribute('rel'));
}

function anchorLinksToTagOrCategoryPage(element) {
    return /\bcategory|tag\b/i.test(element.getAttribute('rel'));
}

function anchorLooksLikeDownloadFlashLink(element) {
    return /^https?:\/\/(www\.|get\.)(adobe|macromedia)\.com\/(((products|[a-zA-Z]{1,2}|)\/flashplayer|flashplayer|go\/getflash(player)?)|(shockwave\/download\/(index|download)\.cgi\?P1_Prod_Version=ShockwaveFlash)\/?$)/i.test(element.href);
}

function elementsHaveSameTagAndClassNames(element1, element2) {
    return element1.tagName === element2.tagName && element1.className === element2.className;
}

/**
 * 根据标签名和 class-name 为元素生成一个 CSS 选择器。
 */
function selectorForElement(element) {
    var selector = element.tagName,
        classList = element.classList,
        classListLength = classList.length;
    for (var i = 0; i < classListLength; i++) {
        selector += '.' + classList[i];
    }
    return selector;
}

/**
 * 根据输入的层级，计算元素的"指纹"，用于标记元素。
 */
function elementFingerprintForDepth(element, depth) {
    function n(currentElement, currentDepth) {
        if (!currentElement) {
            return '';
        }
        var fingerprint = [];
        fingerprint.push(selectorForElement(currentElement));
        var childrenElement = currentElement.children,
            childrenLength = childrenElement.length;
        if (childrenLength && currentDepth > 0) {
            fingerprint.push(slash);
            for (var i = 0; i < childrenLength; i++) {
                fingerprint.push(n(childrenElement[i], currentDepth - 1));
                if (i !== childrenLength - 1) {
                    fingerprint.push(pipe);
                }
            }
            fingerprint.push(doubleBackslash);
        }
        return fingerprint.join('');
    }

    const slash = ' / ',
        doubleBackslash = ' \\',
        pipe = ' | ';
    return n(element, depth);
}

function childrenOfParentElement(element) {
    var parentElement = element.parentElement;
    return parentElement ? parentElement.children : [];
}

/**
 * 根据数组里元素的值按降序重新排列，以元素在数组中的序号和值组成对象数组。
 * @example [1,4,6,1,98] -> [{"4", 98}, {"2", 6}, {"1", 4}, {"0", 1}, {"3", 1}]
 */
function arrayOfKeysAndValuesOfObjectSortedByValueDescending(targetArray) {
    var result = [];
    for (var item in targetArray) {
        if (targetArray.hasOwnProperty(item)) {
            result.push({
                key: item,
                value: targetArray[item]
            });
        }
    }
    result.sort(function (value1, value2) {
        // 降序排列
        return value2.value - value1.value;
    });
    return result;
}

/**
 * 根据输入的层级遍历元素及其子元素，对每个元素执行 callback。
 * @param element 需要被遍历的元素。
 * @param depth 遍历层级。
 * @param callback 每个元素被遍历后的回调。
 */
function walkElementSubtree(element, depth, callback) {
    if (depth >= 0) {
        var childrenElement = element.children;
        for (var i = 0; childrenElement.length > i; i++) {
            walkElementSubtree(childrenElement[i], depth - 1, callback);
        }
        callback(element, depth);
    }
}

function elementIndicatesItIsASchemaDotOrgArticleContainer(element) {
    var itemtype = element.getAttribute("itemtype");
    return /^https?:\/\/schema\.org\/(News)?Article$/.test(itemtype);
}

function cleanStyleAndClassList(element) {
    element.classList.length || element.removeAttribute('class'); // 当元素的 class 属性值为空时，清楚这类空值的属性。
    element.getAttribute('style') || element.removeAttribute('style'); // 当元素的 style 属性值为空时，清楚这类空值的属性。
}

/**
 * 寻找元素中可见的非空格文字。
 * @param element 被寻找的元素。
 * @param maxDepth 在 DOM 中遍历的最大层级。
 * @param maxNumberOfJudgments 判断 DOM 是否符合要求时的最大判断次数，超过这个次数，则不再继续进行寻找。
 * @param isLegacyHTML 元素是否属于 HTML4.0 下的页面。
 * @param ignoreElements 需要被忽略的元素的列表。
 * @returns {Array}
 */
function getVisibleNonWhitespaceTextNodes(element, maxDepth, maxNumberOfJudgments, isLegacyHTML, ignoreElements) {
    function isElementRelatedToSharing(e) {
        var firstChild = e.children[0];
        if (firstChild) {
            var childrenElement = firstChild.children;
            for (var i = 0; i < childrenElement.length; i++) {
                if (getComputedStyle(childrenElement[i])['float'] !== 'none') {
                    return false;
                }
            }
        }
        return true;
    }

    function findTextNodesRecursively(element, currentDepth) {
        if (element.nodeType === Node.TEXT_NODE) {
            if (/\S/.test(element.nodeValue)) {
                results.push(element);
            }
        }
        if (element.nodeType === Node.ELEMENT_NODE &&
            isElementVisible(element) &&
            (!maxNumberOfJudgments || ++numberOfJudgmentsForElementNode <= maxNumberOfJudgments) &&
            (!ignoreElements || !ignoreElements.has(element))) {
            var tagName = element.tagName;
            if ('IFRAME' !== tagName && 'FORM' !== tagName) {
                if (tagNamesToRecursivelySearchForTextNodes[tagName]) {
                    currentDepth--;
                } else if ('UL' !== tagName && 'OL' !== tagName || !isElementRelatedToSharing(element)) {
                    var m = element.parentElement;
                    if (m) {
                        var d = m.tagName;
                        if (d === 'SECTION' && !element.previousElementSibling && !element.nextElementSibling) {
                            currentDepth--;
                        }
                    }
                } else {
                    currentDepth--;
                }
                var childrenDepth = currentDepth + 1;
                if (maxDepth > childrenDepth) {
                    for (var g = element.childNodes, f = g.length, p = 0; f > p; p++) {
                        findTextNodesRecursively(g[p], childrenDepth);
                    }
                }
            }
        }
    }

    var numberOfJudgmentsForElementNode = 0,
        results = [],
        tagNamesToRecursivelySearchForTextNodes = {
            P: 1,
            STRONG: 1,
            B: 1,
            EM: 1,
            I: 1,
            SPAN: 1,
            SECTION: 1
        };

    if (isLegacyHTML) {
        tagNamesToRecursivelySearchForTextNodes.CENTER = 1;
        tagNamesToRecursivelySearchForTextNodes.FONT = 1;
    }
    findTextNodesRecursively(element, 0);
    return results;
}

/**
 * 遍历元素中可见的非空格文字，对每一个 textNode 执行一个回调，最终返回一个以回调函数返回值为 key，textNode 长度为 value 的对象集合。
 * @param element 被遍历的元素。
 * @param callback 对元素中的每一个可见的非空格 textNode 执行的回调。
 * @returns 返回一个对象集合，集合的 key 为回调函数返回值为，value 为 textNode 长度。
 */
function mapOfVisibleTextNodeComputedStyleReductionToNumberOfMatchingCharacters(element, callback) {
    const maxDepth = 100;
    var result = {},
        visibleNonWhitespaceTextNodes = getVisibleNonWhitespaceTextNodes(element, maxDepth);
    for (var o = 0; o < visibleNonWhitespaceTextNodes.length; o++) {
        var textNode = visibleNonWhitespaceTextNodes[o],
            textNodeLength = textNode.length,
            textNodeParent = textNode.parentElement,
            textNodeParentStyle = getComputedStyle(textNodeParent),
            callbackResult = callback(textNodeParentStyle);
        if (result[callbackResult]) {
            result[callbackResult] += textNodeLength;
        } else {
            result[callbackResult] = textNodeLength;
        }
    }
    return result;
}

/**
 * 找出 Dictionary 中最大的 value 对应的 key。
 */
function keyOfMaximumValueInDictionary(dict) {
    var keyOfMaximumValue,
        maximumValue;
    for (var key in dict) {
        var value = dict[key];
        if (!maximumValue || maximumValue < value) {
            keyOfMaximumValue = key;
            maximumValue = value;
        }
    }
    return keyOfMaximumValue;
}

function elementIsProtected(element) {
    return element.classList.contains('protected') || element.querySelector('.protected');
}

/**
 * 找出元素中主要的文字样式信息（字体、字号和颜色）。
 */
function dominantFontAppearanceForElement(element) {
    var fontAppearance = mapOfVisibleTextNodeComputedStyleReductionToNumberOfMatchingCharacters(element, function (e) {
        return e.fontFamily + '|' + e.fontSize + '|' + e.color;
    });
    return keyOfMaximumValueInDictionary(fontAppearance);
}

/**
 * 从元素主要文字样式信息中筛选出主要字号。
 */
function dominantFontSizeInPointsFromFontAppearanceString(info) {
    return info ? parseInt(info.split('|')[1]) : null;
}

/**
 * 从元素主要文字的样式信息中筛选出主要颜色。
 */
function dominantColorInPointsFromFontAppearanceString(info) {
    return info ? info.split('|')[2] : null;
}

/**
 * 如果元素中有文字节点，则依据最长的文字节点，从最靠近该节点的容器中获取文字的颜色。
 */
function dominantStyleFromComputedClosestContainer(element) {

    function getTextNodes(element) {
        let textNodes = [];
        if (element) {
            let nodes = element.childNodes;
            let i = nodes.length;
            while (i--) {
                let node = nodes[i];
                let nodeType = node.nodeType;
                if (nodeType === Node.TEXT_NODE) {
                    textNodes.push(node);
                } else if (nodeType === Node.ELEMENT_NODE || nodeType === Node.DOCUMENT_NODE || nodeType === Node.DOCUMENT_FRAGMENT_NODE) {
                    textNodes = textNodes.concat(getTextNodes(node));
                }
            }
        }
        return textNodes;
    }


    // 此处需要递归取得所有子节点，包括 TEXT_NODE 和 ELEMENT_NODE
    let allChildren = getTextNodes(element);

    if (allChildren.length > 0) {
        let elementWithDominantText,
            depthOfElementWithDominantText = 0;
        for (let i = 0; i < allChildren.length; i++) {
            let child = allChildren[i];

            // 如果不是 ELEMENT_NODE（主要可能是 TEXT_NODE），则取 parentNode 的样式
            // 这样会取到 ParentNode 中第一个字符的样式，但认为可以接受
            if (child.nodeType !== Node.ELEMENT_NODE) {
                child = child.parentElement;
            }
            let depth = depthOfElementWithinElement(child, element);
            if (!elementWithDominantText || (child.textContent.length > element.textContent.length / 2 && depth > depthOfElementWithDominantText)) {
                elementWithDominantText = child;
                depthOfElementWithDominantText = depth;
            }
        }

        return getComputedStyle(elementWithDominantText);
    } else {
        return getComputedStyle(element);
    }
}

function dominantColorFromComputedClosestContainer(element) {
    return dominantStyleFromComputedClosestContainer(element)['color'];
}

function dominantFontWeigetFromComputedClosestContainer(element) {
    return dominantStyleFromComputedClosestContainer(element)['font-weight'];
}

function dominantFontSizeFromComputedClosestContainer(element) {
    return parseInt(dominantStyleFromComputedClosestContainer(element)['font-size']);
}

function dominantTextIsBold(element) {
    var fontWeight = dominantFontWeigetFromComputedClosestContainer(element);
    if (typeof fontWeight === 'number' && fontWeight >= 600) {
        // 兼容 Chrome getComputedStyle 中 font-weight的返回值
        return true;
    }
    if (typeof fontWeight === 'string' && (fontWeight === 'bold' || fontWeight === 'bolder')) {
        // 兼容 Safar getComputedStyle 中 font-weight的返回值i
        return true;
    }
    return false;
}

/**
 * 十六进制转换为 RGB。
 */
function hex2Rgb(hex) {
    var rgb = [];
    if (/^\#[0-9A-F]{3}$/i.test(hex)) {
        // 判断传入是否为三位十六进制数
        var sixHex = '#';
        // 把三位16进制数转化为六位
        hex.replace(/[0-9A-F]/ig, function (kw) {
            sixHex += kw + kw;
        });
        hex = sixHex;
    }
    if (/^#[0-9A-F]{6}$/i.test(hex)) {
        // 判断传入是否为六位十六进制数
        hex.replace(/[0-9A-F]{2}/ig, function (kw) {
            rgb.push(eval('0x' + kw)); // 十六进制转化为十进制并存如数组
        });
        return 'rgb(' + rgb.join(',') + ')'; // 输出RGB格式颜色
    } else {
        return null;
    }
}

/**
 * RGB 转换为 HSL。
 * @param rgb
 * @return {*}
 */
function rgbToHsl(rgb) {
    var rgbArray = rgb.match(/\d{1,3}/g);

    if (rgbArray.length !== 3) {
        return null;
    }

    var r = rgbArray[0],
        g = rgbArray[1],
        b = rgbArray[2];

    r /= 255;
    g /= 255;
    b /= 255;

    var max = Math.max(r, g, b),
        min = Math.min(r, g, b);

    var h,
        s,
        l = (max + min) / 2;

    if (max === min) {
        // achromatic
        h = s = 0;
    } else {
        var d = max - min;
        s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
        switch (max) {
            case r:
                h = (g - b) / d + (g < b ? 6 : 0);
                break;
            case g:
                h = (b - r) / d + 2;
                break;
            case b:
                h = (r - g) / d + 4;
                break;
        }
        h /= 6;
    }

    return 'hsl(' + [h, s, l].join(',') + ')';
}

/**
 * 把 HEX 或 RGB 颜色格式统一转换为 HSL。
 */
function formatColorToHsl(color) {
    if (/^\#([0-9A-F]{3}|[0-9A-F]{6})$/i.test(color)) {
        // 判断是否为格式正确的16进制颜色值，如果是，则先转换为 RGB 格式的颜色值
        color = hex2Rgb(color);
    }
    if (/^rgb\((\d{1,3}\,( ?)){2}\d{1,3}\)$/i.test(color)) {
        // 判断是否为格式正确的 RGB 格式，如果是，则转换为 HSL。
        color = rgbToHsl(color);
    }
    return color;
}

/**
 * 判断两个颜色的深浅。
 * @param color1 用于比较的颜色1。
 * @param color2 用于比较的颜色2.
 * @param strictMode 严格模式。严格模式下仅颜色1与颜色2的色相，饱和度都相同，但颜色1亮度较大时，才会认为颜色1比颜色2更浅。
 * @return {boolean} 颜色1比颜色2更浅（即亮度较大），则返回 true。
 */
function isLighterColor(color1, color2, strictMode) {

    function getHueOfHsl(hsl) {
        var result = hsl.split(',');
        return result[0].replace(')', '');
    }

    function getSaturationOfHsl(hsl) {
        var result = hsl.split(',');
        return result[1].replace(')', '');
    }

    function getLightnessOfHsl(hsl) {
        var result = hsl.split(',');
        return result[2].replace(')', '');
    }

    if (!color1 || !color2) {
        return false;
    }

    var hslColor1 = formatColorToHsl(color1);
    var hslColor2 = formatColorToHsl(color2);

    if (getLightnessOfHsl(hslColor1) > getLightnessOfHsl(hslColor2)) {
        // 数值越大，颜色越浅
        if (strictMode) {
            if (getHueOfHsl(hslColor1) === getHueOfHsl(hslColor2) && getSaturationOfHsl(hslColor1) === getSaturationOfHsl(hslColor2)) {
                return true;
            }
        } else {
            return true;
        }
    }
    return false;
}

function canvasElementHasNoUserVisibleContent(canvasElement) {
    if (!canvasElement.width || !canvasElement.height) {
        return true;
    }
    var canvasContent = canvasElement.getContext('2d'),
        imageData = canvasContent.getImageData(0, 0, canvasElement.width, canvasElement.height).data;

    for (var i = 0; i < imageData.length; i += 4) {
        var dataItem = imageData[i + 3];
        if (dataItem) {
            return false;
        }
    }
    return true;
}

/**
 * 检测 url 是否在白名单内，如果命中白名单，则可以根据特定的 class-name 找到文章内容。
 * @param hostname 需要被检测的 url。
 * @param callback 对命中白名单的 url 执行的回调，回调中需要根据 selector 找到文章的 DOM。
 */
function findArticleNodeSelectorsInWhitelistForHostname(hostname, callback) {
    const whitelistForHostname = [
        [AppleDotComAndSubdomainsRegex, "*[itemprop='articleBody']"],
        [/^(.+\.)?buzzfeed\.com\.?$/, "article #buzz_sub_buzz"],
        [/^(.+\.)?mashable\.com\.?$/, ".parsec-body .parsec-container"],
        [/^(.+\.)?cnet\.com\.?$/, "#rbContent.container"],
        [/^(.+\.)?engadget\.com\.?$/, "main article #page_body"],
        [/^(.*\.)?m\.wikipedia\.org\.?$/, "#content #bodyContent"],
        [/^mp\.weixin\.qq\.com\.?$/, ".rich_media_content"],
        [/^kuaibao\.qq\.com\.?$/, ".content_box"],
    ];
    for (var i = 0; i < whitelistForHostname.length; i++) {
        var whitelistItem = whitelistForHostname[i],
            whitelistHostname = whitelistItem[0];

        if (whitelistHostname.test(hostname.toLowerCase())) {
            console.log("命中白名单找文章", whitelistHostname, hostname.toLowerCase());
            var whitelistSelector = whitelistItem[1],
                elementFound = callback(whitelistSelector);
            if (elementFound) {
                console.log("命中白名单找文章，且已经找到文章");
                return;
            }
        }
    }
}

/**
 * 检测 url 是否在白名单内，白名单内的网站需要保留不可见元素的网站（如某些网站会先隐藏正文的一部分，需要点击按钮阅读更多），这些隐藏元素需要保留。
 */
function functionToPreventPruningDueToInvisibilityInWhitelistForHostname(hostname) {
    const whiteList = [
        [/^mobile\.nytimes\.com\.?$/, function (element, articleNode) {
            var currentElement = element;
            if (!articleNode) {
                return false;
            }
            while (currentElement && currentElement !== articleNode) {
                if (currentElement.classList.contains('hidden')) {
                    return true;
                }
                currentElement = currentElement.parentElement;
            }
            return false;
        }]
    ];

    var whiteListCount = whiteList.length;
    for (var i = 0; i < whiteListCount; ++i) {
        var currentWhiteList = whiteList[i],
            whiteListRegEx = currentWhiteList[0];
        if (whiteListRegEx.test(hostname.toLowerCase())) {
            return currentWhiteList[1];
        }
    }

    return null;
}

/**
 * 判断元素是否为标题
 */
function elementIsAHeader(element) {
    return !!{
        H1: 1,
        H2: 1,
        H3: 1,
        H4: 1,
        H5: 1,
        H6: 1
    }[element.tagName];
}

function leafElementForElementAndDirection(element, treeWalkerKey) {
    var currentDocument = element.ownerDocument,
        treeWalker = currentDocument.createTreeWalker(currentDocument.body, NodeFilter.SHOW_ELEMENT, {
            acceptNode: function (element) {
                return element.children.length === 0 ? NodeFilter.FILTER_ACCEPT : NodeFilter.FILTER_SKIP;
            }
        });
    treeWalker.currentNode = element;
    return treeWalker[treeWalkerKey]();
}

function previousLeafElementForElement(element) {
    return leafElementForElementAndDirection(element, 'previousNode');
}

function nextLeafElementForElement(element) {
    return leafElementForElementAndDirection(element, 'nextNode');
}

function nextNonFloatingVisibleElementSibling(element) {
    var currentElement = element;
    while (currentElement = currentElement.nextElementSibling) {
        if (isElementVisible(currentElement) && getComputedStyle(currentElement)['float'] === 'none') {
            return currentElement;
        }
    }
    return null;
}

/**
 * 在多个元素中找出面积最大的元素。
 */
function elementWithLargestAreaFromElements(elements) {
    var elementLength = elements.length;
    if (!elementLength) {
        return null;
    }
    var result,
        largestArea = 0;
    for (var i = 0; i < elementLength; i++) {
        var element = elements[i],
            elementBoundingRect = cachedElementBoundingRect(element),
            elementArea = elementBoundingRect.width * elementBoundingRect.height;
        if (elementArea > largestArea) {
            result = element;
            largestArea = elementArea;
        }
    }
    return result;
}

function unwrappedArticleContentElement(element) {
    var currentElement = element;
    while (true) {
        var childNodes = currentElement.childNodes,
            childNodesLength = childNodes.length,
            targetChildNode = null;
        for (var a = 0; a < childNodesLength; a++) {
            var childNode = childNodes[a],
                childNodeType = childNode.nodeType,
                isEffectiveNode = function () {
                    return childNodeType === Node.ELEMENT_NODE ? true : childNodeType === Node.TEXT_NODE ? !isNodeWhitespace(childNode) : false;
                }();
            if (isEffectiveNode) {
                if (targetChildNode) {
                    return currentElement;
                }
                var childNodeTagName = childNode.tagName;
                if (childNodeTagName !== 'DIV' && childNodeTagName !== 'ARTICLE' && childNodeTagName !== 'SECTION') {
                    return currentElement;
                }
                targetChildNode = childNode;
            }
        }
        if (!targetChildNode) {
            break;
        }
        currentElement = targetChildNode;
    }
    return currentElement;
}

function elementsMatchingClassesInClassList(classList, contentDocument) {
    return elementsOfSameClassIgnoringClassNamesMatchingRegexp(classList, contentDocument);
}

function elementsMatchingClassesInClassListIgnoringCommonLayoutClassNames(classList, contentDocument) {
    const ignoringClassNames = /clearfix/i;
    return elementsOfSameClassIgnoringClassNamesMatchingRegexp(classList, contentDocument, ignoringClassNames);
}

function elementsMatchingClassesInClassListIgnoringClassesWithNumericSuffix(classList, contentDocument) {
    const ignoringClassNames = /\d+$/;
    return elementsOfSameClassIgnoringClassNamesMatchingRegexp(classList, contentDocument, ignoringClassNames);
}

/**
 * 从指定的 class-name 列表中排除特定的 class-name，然后获取剩下的 class-name 对应的元素。
 * @param classList 需要获取的 class-name 列表。
 * @param contentDocument 一个 document 实例。
 * @param ignoringClassNames 需要排除的特定的 class-name。
 * @returns 返回排除后剩下的 class-name 对应的元素。
 */
function elementsOfSameClassIgnoringClassNamesMatchingRegexp(classList, contentDocument, ignoringClassNames) {
    var selector = '';
    for (var i = 0; i < classList.length; i++) {
        var className = classList[i];
        if (!ignoringClassNames || !ignoringClassNames.test(className)) {
            selector += '.' + className;
        }
    }
    try {
        return contentDocument.querySelectorAll(selector);
    } catch (error) {
        return [];
    }
}

/**
 * 判断 img 元素的 src 是否与其他父元素的 backgroundImage 相同。
 */
function imageIsContainedByContainerWithImageAsBackgroundImage(element) {
    var parentElement = element.parentElement;
    if (!parentElement || !parentElement.style.backgroundImage) {
        return false;
    }
    var backgroundImageValue = /url\((.*)\)/.exec(parentElement.style.backgroundImage);
    if (!backgroundImageValue || backgroundImageValue.length !== 2) {
        return false;
    }
    var backgroundImageUrl = backgroundImageValue[1];
    return element.src === backgroundImageUrl;
}

/**
 * 寻找元素中具有"平行结构"的子元素（具有相同 class-name，并且这类子元素超过总子元素的一半）。
 * @param element 在该元素中寻找具有"平行结构"的子元素。
 * @returns 返回符合"平行结构"条件的子元素，如果找不到符合条件的子元素，则返回空数组。
 */
function childrenWithParallelStructure(element) {
    var elementChildren = element.children;
    if (!elementChildren) {
        return [];
    }
    var elementChildrenLength = elementChildren.length;
    if (!elementChildrenLength) {
        return [];
    }
    var classNameAndElements = {};
    for (var i = 0; elementChildrenLength > i; i++) {
        var elementChild = elementChildren[i];
        if (!CandidateTagNamesToIgnore[elementChild.tagName] && elementChild.className) {
            var elementChildClassList = elementChild.classList;
            for (var j = 0; j < elementChildClassList.length; j++) {
                var className = elementChildClassList[j],
                    targetElements = classNameAndElements[className];
                if (targetElements) {
                    targetElements.push(elementChild);
                } else {
                    classNameAndElements[className] = [elementChild];
                }
            }
        }
    }
    var halfOfElementChildrenLength = Math.floor(elementChildrenLength / 2);
    for (var className in classNameAndElements) {
        var elements = classNameAndElements[className];
        if (elements.length > halfOfElementChildrenLength) {
            return elements;
        }
    }
    return [];
}

function elementAppearsToBeCollapsed(element) {
    return element.getAttribute('aria-expanded') === 'false' && !isElementVisible(element);
}

/**
 * 图片是否是微信公众号图片
 * @param String url
 * @return Boolean 是否是微信公众号图片
 */
function isWeChatMPImage(url) {
    const WeChatMPImageURLRegex = /^https?:\/\/mmbiz\.qpic\.cn/;
    return url.test(WeChatMPImageURLRegex);
}

/**
 * 图片是否是微信公众号内的广告类图片。
 * @param imageElement 图片元素。
 * @return Boolean 是否是微信公众号动图。
 */
function imageAppearsToBeAd(imageElement) {

    if (imageElement.getAttribute('data-type') === 'gif' && imageElement.getAttribute('data-w') >= 700) {
        return true;
    }

    if (imageElement.classList.contains('__bg_gif') && imageElement.getAttribute('data-w') >= 700) {
        return true;
    }

    return false;
}

function audioPlayerWithAudioData(audioData, contentDocument) {
    let audioContainer = contentDocument.createElement('div');
    let audioElement = contentDocument.createElement('audio');
    audioElement.src = audioData.contentURL;
    audioContainer.appendChild(audioElement);

    return audioContainer;
}

/**
 * 候选的文章，选出可能的文章节点后会使用本对象进行封装。
 * @param element 候选文章节点。
 * @param contentDocument 一个 document 实例。
 * @constructor
 */
var CandidateElement = function (element, contentDocument) {
    this.element = element;
    this.contentDocument = contentDocument;
    this.textNodes = this.usableTextNodesInElement(this.element);
    this.rawScore = this.calculateRawScore();
    this.tagNameAndAttributesScoreMultiplier = this.calculateElementTagNameAndAttributesScoreMultiplier();
    this.languageScoreMultiplier = 0;
    this.depthInDocument = 0;
};

/**
 * 如果页面元素可用，则创建一个候选的文章对象。
 * @param element 页面元素。
 * @param contentDocument 一个 document 实例。
 * @param isForSimilarClass 与已有的候选文章是否有相同的 class-name。
 */
CandidateElement.candidateIfElementIsViable = function (element, contentDocument, isForSimilarClass) {
    var elementRect = cachedElementBoundingRect(element);
    if (elementRect.width < CandidateMinimumWidth || elementRect.height < CandidateMinimumHeight) {
        return null;
    }
    if (elementRect.width * elementRect.height < CandidateMinimumArea) {
        return null;
    }
    if (!isForSimilarClass && elementRect.top > CandidateMaximumTop) {
        return null;
    }
    if (CandidateElement.candidateElementAdjustedHeight(element) < CandidateMinimumHeight) {
        return null;
    }
    return new CandidateElement(element, contentDocument);
};

/**
 * 如果页面元素可用，则创建一个额外的候选文章节点（已有文章的上一篇/下一篇文章可能在同一个页面内）。
 * @param element 页面元素。
 * @param selectedArticle 已有的文章。
 * @param contentDocument 一个 document 实例。
 * @param isPrepended 元素是否在已有文章前面。
 */
CandidateElement.extraArticleCandidateIfElementIsViable = function (element, selectedArticle, contentDocument, isPrepended) {
    const InlineTextContainerTagNames = 'a, b, strong, i, em, u, span';
    var elementRect = cachedElementBoundingRect(element),
        selectedArticleRect = cachedElementBoundingRect(selectedArticle.element);
    if ((isPrepended && elementRect.height < PrependedArticleCandidateMinimumHeight || !isPrepended && elementRect.height < AppendedArticleCandidateMinimumHeight) && element.childElementCount && element.querySelectorAll('*').length !== element.querySelectorAll(InlineTextContainerTagNames).length) {
        return null;
    }
    if (isPrepended) {
        if (elementRect.bottom > selectedArticleRect.top) {
            return null;
        }
    } else if (elementRect.top < selectedArticleRect.bottom) {
        return null;
    }
    if (!isPrepended) {
        var articleCandidateVerticalDistanceFromArticle = elementRect.top - selectedArticleRect.bottom;
        if (articleCandidateVerticalDistanceFromArticle > AppendedArticleCandidateMaximumVerticalDistanceFromArticle) {
            return null;
        }
    }
    if (elementRect.left > selectedArticleRect.right || elementRect.right < selectedArticleRect.left) {
        return null;
    }
    if (elementLooksLikePartOfACarousel(element)) {
        return null;
    }
    var result = new CandidateElement(element, contentDocument);
    result.isPrepended = isPrepended;
    return result;
};

/**
 * 为了计算元素内文字的密度，如果先对相关元素的高度进行优化计算，例如表单，以及列表，元素占用的高度通过比较大，但文字内容较少，
 * 需要对这些元素的有效高度进行计算。
 */
CandidateElement.candidateElementAdjustedHeight = function (element) {
    var elementRect = cachedElementBoundingRect(element),
        adjustedHeight = elementRect.height,
        indicatorElements = element.getElementsByTagName('form');

    for (var i = 0; i < indicatorElements.length; i++) {
        var indicatorElement = indicatorElements[i],
            indicatorRect = cachedElementBoundingRect(indicatorElement);
        if (indicatorRect.width > elementRect.width * CandidateMinimumWidthPortionForIndicatorElements) {
            adjustedHeight -= indicatorRect.height;
        }
    }
    var listContainers = element.querySelectorAll('ol, ul'),
        listContainersCount = listContainers.length,
        lastListContainerSubstracted = null;
    for (var i = 0; listContainersCount > i; i++) {
        var listContainer = listContainers[i];
        if (!(lastListContainerSubstracted && lastListContainerSubstracted.compareDocumentPosition(listContainer) & Node.DOCUMENT_POSITION_CONTAINED_BY)) {
            var listItems = listContainer.getElementsByTagName('li'),
                listItemCount = listItems.length,
                listRect = cachedElementBoundingRect(listContainer);
            if (listItemCount) {
                var averageListItemHeight = listRect.height / listItemCount,
                    firstListItemStyle = getComputedStyle(listItems[0]),
                    listLineHeight = parseInt(firstListItemStyle.lineHeight);
                if (isNaN(listLineHeight)) {
                    var listFontSize = fontSizeFromComputedStyle(firstListItemStyle);
                    listLineHeight = listFontSize * BaseLineHeightRatio;
                }
                if (listRect.width > elementRect.width * CandidateMinimumWidthPortionForIndicatorElements && ((averageListItemHeight / listLineHeight) < CandidateMinumumListItemLineCount)) {
                    adjustedHeight -= listRect.height;
                    lastListContainerSubstracted = listContainer;
                }
            } else {
                adjustedHeight -= listRect.height;
            }
        }
    }
    return adjustedHeight;
};

CandidateElement.prototype = {
    // 认为可能是文章正文的候选元素
    calculateRawScore: function () {
        var score = 0;
        var textNodes = this.textNodes;
        for (var i = 0; i < textNodes.length; i++)
            score += this.rawScoreForTextNode(textNodes[i]);
        return score;
    },

    calculateElementTagNameAndAttributesScoreMultiplier: function () {
        return scoreMultiplierForElementTagNameAndAttributes(this.element);
    },

    calculateLanguageScoreMultiplier: function () {
        this.languageScoreMultiplier === 0 && (this.languageScoreMultiplier = languageScoreMultiplierForTextNodes(this.textNodes));
    },

    depth: function () {
        if (!this.depthInDocument) {
            this.depthInDocument = elementDepth(this.element);
        }
        return this.depthInDocument;
    },

    finalScore: function () {
        this.calculateLanguageScoreMultiplier();
        return this.basicScore() * this.languageScoreMultiplier;
    },

    basicScore: function () {
        return this.rawScore * this.tagNameAndAttributesScoreMultiplier;
    },

    scoreDensity: function () {
        var ignoredArea = 0,
            ignoredElement = this.element.querySelector(DensityExcludedElementSelector);

        if (ignoredElement) {
            ignoredArea = ignoredElement.clientWidth * ignoredElement.clientHeight;
        }

        var childrenElements = this.element.children || [];
        var childrenElementsCount = childrenElements.length;

        for (var i = 0; i < childrenElementsCount; i++) {
            var element = childrenElements[i];

            if (elementIsCommentBlock(element)) {
                ignoredArea += element.clientWidth * element.clientHeight;
            }
        }

        var articleArea = cachedElementBoundingRect(this.element).width * cachedElementBoundingRect(this.element).height;
        var maximumImageArea = articleArea * MaximumContentMediaAreaToArticleAreaRatio;
        var imageMinimumWidth = cachedElementBoundingRect(this.element).width * MinimumContentMediaWidthToArticleWidthRatio;
        var imageElements = this.element.querySelectorAll('img, object, video');
        var imageCount = imageElements.length;

        for (var i = 0; imageCount > i; ++i) {
            var m = cachedElementBoundingRect(imageElements[i]);
            if (m.width >= imageMinimumWidth && m.height > MinimumContentMediaHeight) {
                var d = m.width * m.height;
                maximumImageArea > d && (ignoredArea += d)
            }
        }

        var score = this.basicScore();
        var area = articleArea - ignoredArea;
        var numberOfTextNodes = this.textNodes.length;
        var numberOfCountedTextNodes = 0;
        var sumOfFontSizes = 0;

        for (var i = 0; i < numberOfTextNodes; i++) {
            var parentNode = this.textNodes[i].parentNode;
            console.assert(parentNode, "parentNode of this.textNodes[i] cannot be nil");
            if (parentNode) {
                sumOfFontSizes += fontSizeFromComputedStyle(getComputedStyle(parentNode));
                numberOfCountedTextNodes++;
            }
        }

        var averageFontSize = BaseFontSize;
        if (numberOfCountedTextNodes) {
            averageFontSize = sumOfFontSizes /= numberOfCountedTextNodes;
        }
        this.calculateLanguageScoreMultiplier();
        return score / area * 1e3 * (averageFontSize / BaseFontSize) * this.languageScoreMultiplier;
    },

    usableTextNodesInElement: function (element) {
        var textNodes = [];
        if (!element)
            return textNodes;
        const tagNamesToIgnore = {
            A: 1,
            DD: 1,
            DT: 1,
            NOSCRIPT: 1,
            OL: 1,
            OPTION: 1,
            PRE: 1,
            SCRIPT: 1,
            STYLE: 1,
            TD: 1,
            UL: 1,
            IFRAME: 1
        };
        var contentDocument = this.contentDocument;
        var getUsableTextNodes = function (element) {
            const xPathQuery = 'text()|*/text()|*/a/text()|*/li/text()|*/li/p/text()|*/span/text()|*/em/text()|*/i/text()|*/strong/text()|*/b/text()|*/font/text()|blockquote/*/text()|div[count(./p)=count(./*)]/p/text()|div[count(*)=1]/div/p/text()|div[count(*)=1]/div/p/*/text()';
            var xPathResults = contentDocument.evaluate(xPathQuery, element, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null);
            for (var i = 0; i < xPathResults.snapshotLength; i++) {
                var textNode = xPathResults.snapshotItem(i);
                if (!tagNamesToIgnore[textNode.parentNode.tagName] && !textNode._countedTextNode && !isNodeWhitespace(textNode)) {
                    textNode._countedTextNode = true;
                    textNodes.push(textNode);
                }
            }
        };
        getUsableTextNodes(element);
        var children = childrenWithParallelStructure(element);
        for (var i = 0; i < children.length; i++) {
            var child = children[i];
            getUsableTextNodes(child);
        }
        for (var i = 0; i < textNodes.length; i++) {
            delete textNodes[i]._countedTextNode;
        }
        return textNodes;
    },

    addTextNodesFromCandidateElement: function (otherCandidateElement) {
        for (var j = 0; j < this.textNodes.length; ++j) {
            this.textNodes[j].alreadyCounted = true;
        }
        var otherCandidateElementTextNodes = otherCandidateElement.textNodes;
        for (var j = 0; j < otherCandidateElementTextNodes.length; j++) {
            if (!otherCandidateElementTextNodes[j].alreadyCounted) {
                this.textNodes.push(otherCandidateElementTextNodes[j]);
            }
        }
        for (var j = 0; j < this.textNodes.length; j++) {
            this.textNodes[j].alreadyCounted = null;
        }
        this.rawScore = this.calculateRawScore();
    },

    /**
     * 计算一个文字节点的得分。
     */
    rawScoreForTextNode: function (textNode) {
        const TextNodeMinimumLength = 20; // 一个文字节点最少20个字
        if (!textNode) {
            return 0;
        }

        // 如果长度不达标则不算分
        var length = textNode.length;
        if (TextNodeMinimumLength > length) {
            return 0;
        }

        // 如果父级元素不可见则不算分
        var ancestor = textNode.parentNode;
        if (!isElementVisible(ancestor)) {
            return 0;
        }

        for (var multiplier = 1; ancestor && ancestor !== this.element;) {
            multiplier -= .1;
            ancestor = ancestor.parentNode;
        }

        return Math.pow(length * multiplier, TextNodeLengthPower);
    },

    /**
     * 如果候选元素的得分小于某个值，则认为这不是一篇文章。
     */
    shouldDisqualifyDueToScoreDensity: function () {
        return this.scoreDensity() < ArticleMinimumScoreDensity;
    },

    /**
     * 如果一篇文章中 HR 分隔符的密度太大，则认为这不是一篇文章。
     */
    shouldDisqualifyDueToHorizontalRuleDensity: function () {
        var horizontalRules = this.element.getElementsByTagName("hr");
        var numberOfHRs = horizontalRules.length;
        var numberOfHRsToCount = 0;
        var elementRect = cachedElementBoundingRect(this.element);
        var minimumWidthToCount = elementRect.width * .7;

        for (var i = 0; numberOfHRs > i; ++i) {
            if (horizontalRules[i].clientWidth > minimumWidthToCount) {
                numberOfHRsToCount++;
            }
        }

        if (numberOfHRsToCount) {
            var averageDistanceBetweenHRs = elementRect.height / numberOfHRsToCount;
            if (averageDistanceBetweenHRs < MinimumAverageDistanceBetweenHRElements) {
                return true;
            }
        }
        return false;
    },

    /**
     * 如果候选元素出现的标题太多，大于了某个密度，则认为这不是一篇文章。
     */
    shouldDisqualifyDueToHeaderDensity: function () {
        var headerLinksXPathQuery = "(h1|h2|h3|h4|h5|h6|*/h1|*/h2|*/h3|*/h4|*/h5|*/h6)[a[@href]]";
        var headerLinkResults = this.contentDocument.evaluate(headerLinksXPathQuery, this.element, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null);
        var headerLinkResultsCount = headerLinkResults.snapshotLength;
        if (headerLinkResultsCount > 2) {
            var numberOfHeadersToCount = 0;
            var elementRect = cachedElementBoundingRect(this.element);
            var topBottomIgnoreDistance = elementRect.height * PortionOfCandidateHeightToIgnoreForHeaderCheck;
            for (var i = 0; i < headerLinkResultsCount; ++i) {
                var header = headerLinkResults.snapshotItem(i);
                var headerRect = cachedElementBoundingRect(header);

                if (headerRect.top - elementRect.top > topBottomIgnoreDistance && elementRect.bottom - headerRect.bottom > topBottomIgnoreDistance) {
                    numberOfHeadersToCount++;
                }
            }
            var averageDistanceBetweenHeaders = elementRect.height / numberOfHeadersToCount;

            if (averageDistanceBetweenHeaders < MinimumAverageDistanceBetweenHeaderElements) {
                return true;
            }
        }
        return false;
    },

    /**
     * 如果候选元素和其他元素雷同，说明可能是“相关文章推荐”之类的内容，因此判断这不是一篇文章。
     */
    shouldDisqualifyDueToSimilarElements: function (candidateElements) {
        function t(e, t) {
            if (!e || !t)
                return false;
            var n = 1;
            return e.className ? e.className === t.className : elementFingerprintForDepth(e, n) === elementFingerprintForDepth(t, n)
        }

        const n = 'h1, h2, h3, h4, h5, h6';
        var i = function (e) {
                const t = /related-posts/i;
                for (var n = e.parentElement; n && n !== this.contentDocument.body; n = n.parentElement)
                    if (t.test(n.className))
                        return true;
                return false
            }.bind(this),
            r = this.element;
        if ("ARTICLE" === r.parentElement.tagName)
            return false;
        if ("LI" === r.tagName || "DD" === r.tagName)
            for (var a = r.parentNode, o = a.children.length, l = 0; o > l; ++l) {
                var s = a.children[l];
                if (s.tagName === r.tagName && s.className === r.className && s !== r)
                    return true
            }
        var c = r.classList;
        if (c.length || (r = r.parentElement, r && (c = r.classList, c.length || (r = r.parentElement, r && (c = r.classList)))), c.length) {
            candidateElements || (candidateElements = []);
            for (var u = candidateElements.length, l = 0; u > l; ++l)
                candidateElements[l].element.candidateElement = candidateElements[l];
            for (var m = elementsMatchingClassesInClassListIgnoringCommonLayoutClassNames(c, this.contentDocument), d = false, h = elementDepth(r), g = i(r), f = m.length, l = 0; f > l; ++l) {
                var s = m[l];
                if (s !== r && s.parentElement !== r && r.parentElement !== s && isElementVisible(s)) {
                    var p = s.candidateElement;
                    if ((p || (p = new CandidateElement(s, this.contentDocument))) && p.basicScore() * ReaderMinimumAdvantage > this.basicScore()) {
                        if (s.closest("section") && r.closest("section"))
                            return false;
                        if (SectionRegex.test(s.className) && SectionRegex.test(r.className))
                            return false;
                        if (i(s) && !g)
                            return false;
                        if (!d && cachedElementBoundingRect(s).bottom < cachedElementBoundingRect(this.element).top) {
                            d = true;
                            continue
                        }
                        if (t(r.previousElementSibling, s.previousElementSibling) || t(r.nextElementSibling, s.nextElementSibling)) {
                            var E = r.querySelector(n),
                                v = s.querySelector(n);
                            if (E && v && elementsHaveSameTagAndClassNames(E, v))
                                return true;
                            if (E = r.previousElementSibling, v = s.previousElementSibling, E && v && elementIsAHeader(E) && elementIsAHeader(v) && elementsHaveSameTagAndClassNames(E, v))
                                return true
                        }
                        if (elementDepth(s) === h)
                            for (; s.parentElement && r.parentElement && s.parentElement !== r.parentElement;)
                                s = s.parentElement, r = r.parentElement;
                        for (; r.childElementCount <= 1;) {
                            if (!r.childElementCount || !s.childElementCount)
                                return false;
                            if (s.childElementCount > 1)
                                return false;
                            if (r.firstElementChild.tagName !== s.firstElementChild.tagName)
                                return false;
                            r = r.firstElementChild, s = s.firstElementChild
                        }
                        if (s.childElementCount <= 1)
                            return false;
                        var v = s.firstElementChild,
                            N = s.lastElementChild,
                            E = r.firstElementChild,
                            C = r.lastElementChild;
                        if (v.tagName !== E.tagName)
                            return false;
                        if (N.tagName !== C.tagName)
                            return false;
                        var A = v.className,
                            S = N.className,
                            R = E.className,
                            M = N.className,
                            b = M === R ? 2 : 1;
                        if (A.length || R.length) {
                            if (!A.length || !R.length)
                                return false;
                            if (A === R && elementsMatchingClassesInClassList(E.classList, r).length <= b)
                                return true
                        }
                        if (S.length || M.length) {
                            if (!S.length || !M.length)
                                return false;
                            if (S === M && elementsMatchingClassesInClassList(N.classList, r).length <= b)
                                return true
                        }
                        var y = E.clientHeight,
                            x = C.clientHeight;
                        return y && v.clientHeight && x && N.clientHeight ? y === v.clientHeight || x === N.clientHeight : false
                    }
                }
            }
            for (var l = 0; u > l; ++l)
                candidateElements[l].element.candidateElement = null
        }
        return false
    },

    shouldDisqualifyForDeepLinking: function () {
        function e(e) {
            var t = e.pathname.substring(1).split('/');
            return t[t.length - 1] || t.pop(), t;
        }

        const t = 5;
        for (var n = this.element, i = this.contentDocument.location, r = e(i), a = r.length, o = [], l = n.getElementsByTagName('a'), s = l.length, c = 0; s > c; c++) {
            var u = l[c];
            if (i.host === u.host && !(e(u).length <= a || 0 !== (u.host + u.pathname).indexOf(i.host + i.pathname) || anchorLinksToAttachment(u) || (o.push(u), o.length < t))) {
                var m = n.offsetTop + n.offsetHeight / t;
                return o[0].offsetTop < m;
            }
        }
        return false;
    }
};

/**
 * 寻找文章节点的类，其主要工作流程如下：
 *
 * 搜寻文章正文节点 {@link cookedText}。
 * 搜寻文章标题 {@link postTitle}。
 * 搜寻文章 Meta 信息 {@link cookedMetadataBlock}。
 * 优化正文、标题、Meta 的节点（清理无用节点，优化结构，加强样式） {@link cleanArticleNode}。
 */
ReaderArticleFinder = function (document) {
    this.contentDocument = document;
    this.didSearchForArticleNode = false;
    this.article = null;
    this.didSearchForExtraArticleNode = false;
    this.extraArticle = null;
    this.leadingImage = null;
    this._cachedScrollY = 0;
    this._cachedScrollX = 0;
    this._elementsWithCachedBoundingRects = [];
    this._cachedContentTextStyle = null;
    this.pageNumber = 1;
    this.prefixWithDateForNextPageURL = null;
    this.previouslyDiscoveredPageURLStrings = [];
};

ReaderArticleFinder.prototype = {
    // 判断这个页面有没有文章
    isReaderModeAvailable: function () {
        if (this.findArticleBySearchingWhitelist()) {
            return true;
        }

        this.cacheWindowScrollPosition();

        if (this.findArticleFromMetadata(FindArticleMode.ExistenceOfElement)) {
            return true;
        }

        this.article = this.findArticleByVisualExamination();

        if (this.article) {
            this.articleIsLTR();
        }

        return !!this.article;
    },

    prepareToTransitionToReader: function () {
        this.cookedText(true);
        this.nextPageURL();
        this.articleIsLTR();
    },

    nextPageURL: function () {
        if (!this._nextPageURL) {
            var nextPageURLString = this.nextPageURLString();
            if (typeof ReaderArticleFinderJSController !== 'undefined' && nextPageURLString) {
                nextPageURLString = ReaderArticleFinderJSController.substituteURLForNextPageURL(nextPageURLString);
            }
            this._nextPageURL = nextPageURLString;
        }
        return this._nextPageURL;
    },

    /**
     * 根据当前的文章节点寻找更多的文章容器（判断与当前文章节点是否有类似的 id）。
     */
    containerElementsForMultiPageContent: function () {
        const containerElementIDRegex = /(.*page[^0-9]*|.*article.*item[^0-9]*)(\d{1,2})(.*)/i,
            depthLimited = 3;

        var articleNodeIDs,
            container = [],
            articleNode = this.articleNode(),
            currentDepth = 0;

        while (true) {
            if (articleNodeIDs = containerElementIDRegex.exec(articleNode.getAttribute('id'))) {
                break;
            }
            articleNode = articleNode.parentElement;
            if (!articleNode || currentDepth++ === depthLimited) {
                return [];
            }
        }
        var articleNodeSiblings = childrenOfParentElement(articleNode);
        for (var i = 0; i < articleNodeSiblings.length; i++) {
            var sibling = articleNodeSiblings[i];
            if (sibling !== articleNode) {
                var siblingIDs = containerElementIDRegex.exec(sibling.getAttribute('id'));
                if (siblingIDs && siblingIDs[1] === articleNodeIDs[1] && siblingIDs[3] === articleNodeIDs[3]) {
                    if (!isElementVisible(sibling) || isElementPositionedOffScreen(sibling)) {
                        container.push(sibling);
                    }
                }

            }
        }
        return container;
    },

    cookedMultiPageContentElements: function () {
        return this.containerElementsForMultiPageContent().map(function (element) {
            return this.cleanArticleNode(element, element.cloneNode(true), CleaningType.MainArticleContent, false);
        }, this);
    },

    /**
     * 判断当前 className 是否对找文章具有重要意义（如果 className 包含 small / left / right 等装饰性词汇，则认为没有重要意义）。
     */
    classNameIsSignificantInRouteComputation: function (className) {
        if (!className) {
            return false;
        }
        return !(className.toLowerCase() in StylisticClassNames);
    },

    /**
     * 在计算文章路径时直接忽略的标签。
     */
    shouldIgnoreInRouteComputation: function (element) {
        if (element.tagName === 'SCRIPT' || element.tagName === 'LINK' || element.tagName === 'STYLE') {
            return true;
        }
        if (element.tagName !== 'TR') {
            return false;
        }
        if (element.offsetHeight) {
            return false;
        }
        return true;
    },

    /**
     * 寻找走到文章节点的路径。
     */
    routeToArticleNode: function () {
        for (var hint = [], currentElement = this.articleNode(); currentElement;) {
            var step = {};
            step.tagName = currentElement.tagName;
            var currentElementId = currentElement.getAttribute('id');

            if (currentElement) {
                step.id = currentElementId;

                if (this.classNameIsSignificantInRouteComputation(currentElement.className)) {
                    step.className = currentElement.className;
                }

                step.index = 1;
            }

            for (var sibling = currentElement.previousElementSibling; sibling; sibling = sibling.previousElementSibling) {
                if (!this.shouldIgnoreInRouteComputation(sibling)) {
                    step.index++;
                }
            }

            hint.unshift(step);
            currentElement = currentElement.parentElement;
        }
        return hint;
    },

    /**
     * 判断是否需要将 Article Node 向上调整到父级。
     */
    adjustArticleNodeUpwardIfNecessary: function () {
        if (this.article) {
            var articleNode = this.article.element;
            while (articleNode) {
                if (VeryPositiveClassNameRegEx.test(articleNode.className)) {
                    return void(this.article.element = articleNode);
                }
                articleNode = articleNode.parentElement;
            }

            articleNode = this.article.element;
            if (articleNode.tagName === 'HEADER' && articleNode.parentElement.tagName === 'ARTICLE') {
                return void(this.article.element = articleNode.parentElement);
            }
            var articleNodePreviousElementSibling = articleNode.previousElementSibling;
            if (articleNodePreviousElementSibling && articleNodePreviousElementSibling.tagName === 'FIGURE' && articleNode.parentElement.tagName === 'ARTICLE') {
                return void(this.article.element = articleNode.parentElement);
            }
            var nearestAncestorWithTagNameAboutArticle = articleNode.tagName === 'SECTION' ? articleNode : nearestAncestorElementWithTagName(articleNode, 'SECTION', ['ARTICLE']);
            if (nearestAncestorWithTagNameAboutArticle) {
                var parentElement = nearestAncestorWithTagNameAboutArticle.parentElement,
                    isParentElementContainElementAboutSection = function () {
                        var children = parentElement.children;
                        for (var r = 0; r < children.length; r++) {
                            var child = children[r],
                                childTagName = child.tagName;
                            if (child !== nearestAncestorWithTagNameAboutArticle && (childTagName === 'SECTION' || childTagName === 'HEADER')) {

                                return true;
                            }
                        }
                        return false;
                    }();
                if (isParentElementContainElementAboutSection && (/\barticleBody\b/.test(parentElement.getAttribute('itemprop')) || parentElement.tagName === 'MAIN' || parentElement.getAttribute('role') === 'main' || parentElement.tagName === 'ARTICLE' || parentElement === this.contentDocument.body)) {
                    return void(this.article.element = parentElement);
                }
            }
            const introRegex = /intro/i,
                contentRegex = /body|content/i;
            articleNode = this.article.element;
            if (introRegex.test(articleNode.className) && articleNode.nextElementSibling && contentRegex.test(articleNode.nextElementSibling.className) || contentRegex.test(articleNode.className) && articleNode.previousElementSibling && introRegex.test(articleNode.previousElementSibling.className)) {
                return void(this.article.element = articleNode.parentElement);
            }
            if (articleNode.tagName !== 'ARTICLE') {
                var closestArticleBody = articleNode.parentElement.closest("*[itemprop='articleBody']");
                if (closestArticleBody && closestArticleBody.parentElement.closest(SchemaDotOrgArticleContainerSelector))
                    return void(this.article.element = closestArticleBody)
            }
            var closestArticle = articleNode.closest("article");
            if (closestArticle) {
                articleNode = unwrappedArticleContentElement(articleNode);
                var c = elementDepth(articleNode);
                "P" !== articleNode.tagName || articleNode.className || (articleNode = articleNode.parentElement, c--);
                var m = elementsMatchingClassesInClassListIgnoringCommonLayoutClassNames(articleNode.classList, this.contentDocument);
                1 === m.length && (m = elementsMatchingClassesInClassListIgnoringClassesWithNumericSuffix(articleNode.classList, this.contentDocument));
                for (var d = m.length, h = 0; d > h; ++h) {
                    var u = m[h];
                    if (articleNode !== u && c === elementDepth(u) && isElementVisible(u) && !u.querySelector("article") && dominantFontAppearanceForElement(articleNode) === dominantFontAppearanceForElement(u))
                        return void(this.article.element = closestArticle)
                }
            }
            if (articleNode = this.article.element, !articleNode.getAttribute("id") && articleNode.className) {
                var g = articleNode.tagName,
                    f = articleNode.className,
                    p = articleNode.parentElement;
                if (p)
                    for (var v = p.children, h = 0, E = v.length; E > h; ++h) {
                        var N = v[h];
                        if (N !== articleNode && N.tagName === g && N.className === f) {
                            var S = CandidateElement.candidateIfElementIsViable(N, this.contentDocument, true);
                            if (S && !(S.finalScore() < ReaderMinimumScore))
                                return void(this.article.element = p)
                        }
                    }
            }
        }
    },

    /**
     * 白名单找正文节点。
     */
    findArticleBySearchingWhitelist: function () {
        var articleElement;
        var document = this.contentDocument;
        console.log('findArticleBySearchingWhitelist' + document.location.hostname)

        findArticleNodeSelectorsInWhitelistForHostname(document.location.hostname, function (query) {
            var queryResults = document.querySelectorAll(query);

            if (queryResults.length === 1) {
                articleElement = new CandidateElement(queryResults[0], document);
                return true;
            }
        });

        return articleElement;
    },

    /**
     * 通过五个步骤，逐一寻找文章正文节点。具体步骤说明请浏览 {@link cookedText}。
     * @param forceFindingArticle 是否强制寻找文章（如果为 true，那么按正常逻辑无法找到文章时，则把最可能的结果直接输出）
     */
    articleNode: function (forceFindingArticle) {
        if (!this.didSearchForArticleNode) {
            this.article = this.findArticleBySearchingWhitelist();
        }

        if (!this.article) {
            this.article = this.findArticleBySearchingAllElements();
        }

        if (!this.article) {
            this.article = this.findArticleByVisualExamination();
        }

        if (!this.article) {
            this.article = this.findArticleFromMetadata();
        }

        if (!this.article && forceFindingArticle) {
            this.article = this.findArticleBySearchingAllElements(true);
        }

        this.adjustArticleNodeUpwardIfNecessary();

        if (this.article) {
            this.article.element = unwrappedArticleContentElement(this.article.element)
        }

        this.didSearchForArticleNode = true;

        if (this.article) {
            // 处理 LTR 方向的文章
            this.articleIsLTR();
        }

        if (this.article) {
            return this.article.element;
        }

        return null;
    },

    /**
     * 获取额外的文章节点（已有文章的上一篇/下一篇文章可能在同一个页面内）。
     */
    extraArticleNode: function () {
        if (!this.didSearchForArticleNode) {
            this.articleNode();
        }

        if (!this.didSearchForExtraArticleNode) {
            this.extraArticle = this.findExtraArticle();
            this.didSearchForExtraArticleNode = true;
        }

        if (this.extraArticle) {
            return this.extraArticle.element;
        }

        return null;
    },

    cacheWindowScrollPosition: function () {
        this._cachedScrollY = window.scrollY;
        this._cachedScrollX = window.scrollX;
    },

    contentTextStyle: function () {
        if (this._cachedContentTextStyle)
            return this._cachedContentTextStyle;
        this._cachedContentTextStyle = contentTextStyleForNode(this.contentDocument, this.articleNode(), false);
        if (!this._cachedContentTextStyle)
            this._cachedContentTextStyle = getComputedStyle(this.articleNode());
        return this._cachedContentTextStyle;
    },

    commaCountIsLessThan: function commaCountIsLessThan(node, limit) {
        var count = 0;
        var textContent = node.textContent;
        var i = -1;
        while (count < limit && (i = textContent.indexOf(',', i + 1)) >= 0) {
            count++;
        }
        return count < limit;
    },

    calculateLinkDensityForPruningElement: function (element, t) {
        var textLength = removeWhitespace(element.textContent).length;
        if (!textLength)
            return 0;

        var i = this.article.element;
        var r = function () {
            for (var t = element.originalElement; t && t !== i; t = t.parentElement)
                if (getComputedStyle(t)["float"] !== "none")
                    return t;
            return null;
        }();
        for (var a = element.getElementsByTagName("a"), linkCharacterCount = 0, o = a.length, s = 0; o > s; ++s) {
            var c = a[s];
            !r && c.href && t && t === dominantFontAppearanceForElement(c.originalElement) || (linkCharacterCount += removeWhitespace(c.textContent).length)
        }
        return linkCharacterCount / textLength;
    },

    /**
     * 判断是否需要去掉某个元素。
     */
    shouldPruneElement: function (element, originalElement, fontStyle) {
        const MaxInputToParagraphRatio = .33;
        const MaxPositiveWeightLinkDensity = .5;
        const MaxStandardLinkDensity = .2;
        const MinimumTextLength = 25;
        const MinimumAverageImageArea = 200 * 200;
        var isMPArticle = MPDomainRegex.test(originalElement.ownerDocument.location.hostname);
        var tagName = element.tagName;

        if (!element.parentElement) {
            return false;
        }

        if (originalElement.classList.contains('footnotes')) {
            return false;
        }

        if (element.parentElement.tagName && element.querySelector('img') === 'FIGURE') {
            return false;
        }

        if (tagName === 'IFRAME') {
            return shouldPruneIframe(element, this.contentDocument);
        }

        if (tagName !== 'OBJECT' && tagName !== 'EMBED' && tagName !== 'CANVAS') {
            var hasElementOrTextNodeChild = false;
            var childrenNodesCount = element.childNodes.length;

            for (var i = 0; childrenNodesCount > i; ++i) {
                var node = element.childNodes[i];
                var nodeType = node.nodeType;
                if (nodeType === Node.ELEMENT_NODE || (nodeType === Node.TEXT_NODE && !isNodeWhitespace(node))) {
                    hasElementOrTextNodeChild = true;
                    break;
                }
            }

            if (!hasElementOrTextNodeChild) {
                if ('P' === tagName) {
                    var previousSibling = element.previousSibling;
                    var nextSibling = element.nextSibling;

                    // 如果当前元素的前后都是文本，则认为当前元素需要保留（可能是作者故意留空）
                    if (previousSibling && previousSibling.nodeType === Node.TEXT_NODE && !isNodeWhitespace(previousSibling) &&
                        nextSibling && nextSibling.nodeType === Node.TEXT_NODE && !isNodeWhitespace(nextSibling)) {
                        return false;
                    }
                }

                console.log('prunedElement 去掉空格');
                return true;
            }

            if ('P' === tagName) {
                return false;
            }
        }

        if ('CANVAS' === tagName) {
            // 如果是一个全屏 Canvas，就去掉
            if (window.innerWidth === originalElement.width && window.innerHeight === originalElement.height) {
                return true;
            }

            const ProgressiveClassName = /progressive/i;

            // 如果 Canvas 有 progressive class-name，且下一个相邻元素是 img，则去掉（说明是 placeholder）
            if (ProgressiveClassName.test(originalElement.className) && 'IMG' === originalElement.nextElementSibling.tagName) {
                return true;
            }

            if (canvasElementHasNoUserVisibleContent(originalElement)) {
                return true;
            }

            // 如果这个 Canvas 的父节点叫 CUFON（使用 Canvas 绘制特殊字体的框架），说明是绘制特殊字体用的，则去掉
            if ('CUFON' === element.parentNode.tagName) {
                return true;
            }

            return false;
        }

        // 如果这个标签是 picture，且在 figure 标签内，则放行
        if (element.closest('figure') && element.querySelector('picture')) {
            return false;
        }

        // 命中指定的 class-name 和 ID 的判断，筛选掉一批
        var classIdWeight = 0;
        if (originalElement) {
            if (VeryNegativeClassNameRegEx.test(originalElement.className)) {
                return true;
            }

            var className = originalElement.className;
            var id = originalElement.getAttribute('id');

            if (PositiveRegEx.test(className))
                classIdWeight++;

            if (PositiveRegEx.test(id))
                classIdWeight++;

            if (NegativeRegEx.test(className))
                classIdWeight--;

            if (NegativeRegEx.test(id))
                classIdWeight--;
        }

        if (classIdWeight < 0) {
            console.log('prunedElement 权重 < 0');
            return true;
        }

        // 对 twitter 嵌入做的保护
        if (elementIsProtected(element) || element.querySelector('.tweet-wrapper')) {
            return false;
        }

        // 判断 UL 和 OL 是不是分享按钮们
        if (tagName === 'UL' || tagName === 'OL') {
            if (originalElement.querySelector('iframe') && originalElement.querySelector('script')) {
                return true;
            }

            var childrenElements = originalElement.children;
            var childrenElementsCount = childrenElements.length;
            if (!childrenElementsCount)
                return true;

            var sharingItemsCount = 0;
            var childrenSharingItemsCount = 0;

            for (var i = 0; i < childrenElementsCount; ++i) {
                var childElement = childrenElements[i];
                if (SharingRegex.test(childElement.className))
                    sharingItemsCount++;
                else {
                    var childElementChildren = childElement.children,
                        childElementChildrenCount = childElementChildren.length;

                    if (childElementChildrenCount === 1 && SharingRegex.test(childElementChildren[0].className)) {
                        childrenSharingItemsCount++;
                    }
                }
            }

            if (sharingItemsCount / childrenElementsCount >= MinimumRatioOfListItemsBeingRelatedToSharingToPruneEntireList) {
                return true;
            }

            if (childrenSharingItemsCount / childrenElementsCount >= MinimumRatioOfListItemsBeingRelatedToSharingToPruneEntireList) {
                return true;
            }
        }

        // 判断 OBJECT 对象是不是命中了允许的内嵌视频白名单
        if (tagName === 'OBJECT') {
            var embedElement = element.querySelector('embed[src]'),
                EmbedElementAnchor = embedElement ? anchorForURL(embedElement.src, this.contentDocument) : null;
            if (EmbedElementAnchor && hostnameMatchesHostKnownToContainEmbeddableMedia(EmbedElementAnchor.hostname)) {
                return false;
            }
            var elementData = element.getAttribute('data');
            EmbedElementAnchor = elementData ? anchorForURL(elementData, this.contentDocument) : null;
            return !EmbedElementAnchor || !hostnameMatchesHostKnownToContainEmbeddableMedia(EmbedElementAnchor.hostname)
        }

        if (element.childElementCount === 1) {
            var onlyChildElement = element.firstElementChild;
            if (onlyChildElement.tagName === 'A') {
                return false;
            }
            if (onlyChildElement.tagName === 'SPAN' && onlyChildElement.className === 'converted-anchor' && nearestAncestorElementWithTagName(onlyChildElement, 'TABLE')) {
                return false;
            }
        }

        // 图片可见且图片面积大于最小面积
        var imageElements = element.getElementsByTagName('img');
        var imageElementCount = imageElements.length;
        if (imageElementCount) {
            var averageImageArea = 0;
            for (var i = 0; imageElementCount > i; i++) {
                var originalImage = imageElements[i].originalElement;
                if (isElementVisible(originalImage)) {
                    var originalImageRect = cachedElementBoundingRect(originalImage);
                    averageImageArea += originalImageRect.width / imageElementCount * (originalImageRect.height / imageElementCount)
                }
            }
            if (averageImageArea > MinimumAverageImageArea) {
                return false;
            }
        }

        // 至少有 10 个逗号，否则去掉
        if (!this.commaCountIsLessThan(element, 10)) {
            return false;
        }

        // 节点中图片节点的数量如果多于段落 + 换行 / 2 节点的数量，则去掉
        // 公众号文章经常有图片拼接的文章，不需要走这个规则
        var p = element.getElementsByTagName('p').length;
        var br = element.getElementsByTagName('br').length;
        var numParagraphs = p + Math.floor(br / 2);
        if (imageElementCount > numParagraphs && !isMPArticle) {
            console.log('prunedElement 图片数量大于 < 0', isMPArticle, originalElement.ownerDocument.location.hostname);
            return true;
        }

        var closestTableElement = element.closest('table');
        if (!closestTableElement) {
            if (element.getElementsByTagName('li').length > numParagraphs &&
                dominantFontAppearanceForElement(originalElement.querySelector('li')) !== fontStyle) {
                return true;
            }
            if (element.textContent.length < MinimumTextLength && 1 !== imageElementCount && !isMPArticle) {
                console.log('prunedElement 文字太少且只有一张图', element.textContent.length, MinimumTextLength);
                return true;
            }
            var linkDensity = this.calculateLinkDensityForPruningElement(element, fontStyle);
            if (classIdWeight >= 1 && linkDensity > MaxPositiveWeightLinkDensity) {
                return true;
            }
            if (1 > classIdWeight && linkDensity > MaxStandardLinkDensity) {
                return true;
            }
        }


        if (element.getElementsByTagName('input').length / numParagraphs > MaxInputToParagraphRatio) {
            return true;
        }

        if (element.querySelector('embed')) {
            return true;
        }

        if (tagName === 'TABLE') {
            var textLength = removeWhitespace(element.innerText).length;
            var originalTextLength = removeWhitespace(originalElement.innerText).length;

            if (textLength < (originalTextLength * 0.5)) {
                return true;
            }

            if (this.isMediaWikiPage() && originalElement.classList.contains('toc')) {
                return true;
            }
        }

        return false;
    },

    wordCountIsLessThan: function wordCountIsLessThan(node, limit) {
        var count = 0;
        var textContent = node.textContent;
        var i = -1;
        while ((i = textContent.indexOf(' ', i + 1)) >= 0 && count < limit) {
            count++;
        }
        return count < limit;
    },

    leadingImageIsAppropriateWidth: function (image) {
        if (!this.article || !image) {
            return false;
        }
        return image.getBoundingClientRect().width >= this.article.element.getBoundingClientRect().width - ToleranceForLeadingImageWidthToArticleWidthForFullWidthPresentation
    },

    newDivFromNode: function newDivFromNode(node) {
        var div = this.contentDocument.createElement("div");
        if (node)
            div.innerHTML = node.innerHTML;
        return div;
    },

    headerElement: function () {
        if (!this.article) {
            return null;
        }
        var previousElementOfArticle = this.article.element.previousElementSibling;
        if (previousElementOfArticle && previousElementOfArticle.tagName === 'HEADER') {
            return previousElementOfArticle;
        }
        var articleTitleElement = this._articleTitleElement;
        if (!articleTitleElement) {
            return null;
        }
        var articleTitleElementParentElement = articleTitleElement.parentElement;
        if (articleTitleElementParentElement && articleTitleElementParentElement.tagName === 'HEADER' && !this.article.element.contains(articleTitleElementParentElement)) {
            var imgElements = articleTitleElementParentElement.querySelectorAll('img');
            for (var i = 0; i < imgElements.length; i++) {
                var imgElement = imgElements[i],
                    imgElementRect = cachedElementBoundingRect(imgElement);
                if (imgElementRect.width >= MainImageMinimumWidthAndHeight && imgElementRect.height >= MainImageMinimumWidthAndHeight) {
                    return articleTitleElementParentElement;
                }
            }
        }
        return null;
    },

    cookedLeadingImage: function () {
        const LeadingImageMaximumContainerChildren = 5;
        const LeadingImageCreditRegex = /credit/;
        const LeadingImageCaptionRegex = /caption/;
        const LeadingImageAttributeToKeepRegex = /src|alt/;
        if (!this.article || !this.leadingImage || !this.leadingImageIsAppropriateWidth(this.leadingImage))
            return null;
        var leadingImageContainer = this.leadingImage.closest("figure");
        if (leadingImageContainer) {
            return this.cleanArticleNode(leadingImageContainer, leadingImageContainer.cloneNode(true), CleaningType.LeadingFigure, true);
        }
        var leadingImageContainer = this.leadingImage.parentNode;
        var originalCredit = null;
        var originalCaption = null;
        var numChildren = leadingImageContainer.children.length;
        if (leadingImageContainer.tagName === 'DIV' && numChildren > 1 && LeadingImageMaximumContainerChildren > numChildren) {
            var texts = leadingImageContainer.cloneNode(true).querySelectorAll('p, div');
            for (var length = texts.length, i = 0; length > i; ++i) {
                var text = texts[i];

                if (LeadingImageCreditRegex.test(text.className)) {
                    originalCredit = text.cloneNode(true);
                } else if (LeadingImageCaptionRegex.test(text.className)) {
                    originalCaption = text.cloneNode(true);
                }
            }
        }
        var image = this.leadingImage.cloneNode(false);
        var attributes = image.attributes;
        for (var i = 0; i < attributes.length; ++i) {
            var attributeName = attributes[i].nodeName;
            if (!LeadingImageAttributeToKeepRegex.test(attributeName)) {
                image.removeAttribute(attributeName);
                i--;
            }
        }

        var container = this.contentDocument.createElement('div');
        container.className = 'leading-image';
        container.appendChild(image);

        if (originalCredit) {
            var credit = this.newDivFromNode(originalCredit);
            credit.className = 'credit';
            container.appendChild(credit);
        }
        if (originalCaption) {
            var caption = this.newDivFromNode(originalCaption);
            caption.className = 'caption';
            container.appendChild(caption);
        }
        return container;
    },

    articleBoundingRect: function () {
        if (!this._articleBoundingRect) {
            this._articleBoundingRect = cachedElementBoundingRect(this.article.element);
        }
        return this._articleBoundingRect;
    },

    /**
     * 搜寻文章正文节点，根据下面五个方法，逐一寻找文章正文节点：
     * 白名单找正文节点。{@link findArticleBySearchingWhitelist}
     * 遍历节点找正文。{@link findArticleBySearchingAllElements}
     * 视觉坐标找正文。{@link findArticleByVisualExamination}
     * 根据 HTML Meta 信息找正文。{@link findArticleFromMetadata}
     * 遍历所有节点强制找正文。{@link findArticleBySearchingAllElements}
     *
     * @param forceFindingArticle 是否强制寻找文章（如果为 true，那么按正常逻辑无法找到文章时，则把最可能的结果直接输出）
     */
    cookedText: function (forceFindingArticle) {
        if (this._cookedArticle) {
            return this._cookedArticle.cloneNode(true).outerHTML;
        }
        clearCachedElementBoundingRects();
        this.cacheWindowScrollPosition();

        var rootElement = this.articleNode(forceFindingArticle);
        this._cookedArticle = rootElement ? rootElement.cloneNode(true) : null

        if (this._cookedArticle != null) {
            return this._cookedArticle.outerHTML;
        }

        this._cookedArticle = this.cleanArticleNode(rootElement, this._cookedArticle, CleaningType.MainArticleContent, false);

        if (this._cookedArticle.tagName === 'P') {
            var singleParagraphContainer = document.createElement('div');
            singleParagraphContainer.appendChild(this._cookedArticle);
            this._cookedArticle = singleParagraphContainer;
        }

        var extraArticle = this.extraArticleNode();
        if (extraArticle) {
            var cleanedExtraNode = this.cleanArticleNode(extraArticle, extraArticle.cloneNode(true), CleaningType.MainArticleContent, true);
            if (cleanedExtraNode) {
                if (this.extraArticle.isPrepended) {
                    this._cookedArticle.insertBefore(cleanedExtraNode, this._cookedArticle.firstChild);
                } else {
                    this._cookedArticle.appendChild(cleanedExtraNode);
                }
            }

            var articleBoundingRect = cachedElementBoundingRect(this.article.element);
            var extraArticleBoundingRect = cachedElementBoundingRect(this.extraArticle.element);
            var combinedBoundingRect = {
                top: Math.min(articleBoundingRect.top, extraArticleBoundingRect.top),
                right: Math.max(articleBoundingRect.right, extraArticleBoundingRect.right),
                bottom: Math.max(articleBoundingRect.bottom, extraArticleBoundingRect.bottom),
                left: Math.min(articleBoundingRect.left, extraArticleBoundingRect.left)
            };
            combinedBoundingRect.width = combinedBoundingRect.right - combinedBoundingRect.left;
            combinedBoundingRect.height = combinedBoundingRect.bottom - combinedBoundingRect.top;
            this._articleBoundingRect = combinedBoundingRect;
        }

        this._articleTextContent = this._cookedArticle.innerText;
        var articleHeader = this.headerElement();
        if (this.leadingImage && (!articleHeader || !articleHeader.contains(this.leadingImage))) {
            var leadingImage = this.cookedLeadingImage();
            if (leadingImage) {
                this._cookedArticle.insertBefore(leadingImage, this._cookedArticle.firstChild);
            }
        }
        var hasArticleHeader = !!articleHeader;
        if (hasArticleHeader && extraArticle) {
            if (extraArticle === articleHeader) {
                hasArticleHeader = false;
            }
            if (hasArticleHeader) {
                var documentPosition = extraArticle.compareDocumentPosition(articleHeader);
                if (documentPosition & Node.DOCUMENT_POSITION_CONTAINS || documentPosition & Node.DOCUMENT_POSITION_CONTAINED_BY) {
                    hasArticleHeader = false;
                }
            }
        }
        if (hasArticleHeader) {
            var cleanedArticleHeader = this.cleanArticleNode(articleHeader, articleHeader.cloneNode(true), CleaningType.MainArticleContent, true);
            if (cleanedArticleHeader) {
                this._cookedArticle.insertBefore(cleanedArticleHeader, this._cookedArticle.firstChild);
            }
        }

        return this._cookedArticle.outerHTML;
    },

    elementPinToEdge: function (element) {
        const auxiliaryTag = {
                AREA: 1,
                BR: 1,
                CANVAS: 1,
                EMBED: 1,
                FRAME: 1,
                HR: 1,
                IMG: 1,
                INPUT: 1
            },
            n = 120;
        if (window.scrollY < n) {
            return null;
        }
        var elementRect = cachedElementBoundingRect(element),
            elementInCenterHorizontalAndTop = element.ownerDocument.elementFromPoint((elementRect.left + elementRect.right) / 2, 0); // 获取在 element 水平中间并且与 element 顶部对齐的元素 elementInCenterHorizontalAndTop
        if (elementInCenterHorizontalAndTop && elementInCenterHorizontalAndTop.tagName in auxiliaryTag) {
            // 遍历 elementInCenterHorizontalAndTop，排除特定的一些辅助元素
            elementInCenterHorizontalAndTop = elementInCenterHorizontalAndTop.parentElement;
        }
        var currentElement = elementInCenterHorizontalAndTop;
        // 继续遍历 elementInCenterHorizontalAndTop，如果它是 element 的子元素，则为目标结果
        while (currentElement && currentElement !== element) {
            currentElement = currentElement.parentNode;
        }
        return currentElement ? elementInCenterHorizontalAndTop : null;
    },

    dominantContentSelectorAndDepth: function (element) {
        const depth = 2;
        var selectorAndElement = {},
            selectorAndElementCount = {};

        walkElementSubtree(element, depth, function (currentElement, currentDepth) {
            // 遍历所有子元素，获取元素选择器、对应的元素数量等数据
            if (isElementVisible(currentElement)) {
                var selector = selectorForElement(currentElement) + ' | ' + currentDepth;
                if (selectorAndElementCount[selector]) {
                    selectorAndElementCount[selector] += 1;
                } else {
                    selectorAndElementCount[selector] = 1;
                    selectorAndElement[selector] = currentElement;
                }
            }
        });
        var selectorWithMostElements,
            selectorAndElementCountSortByValueDescending = arrayOfKeysAndValuesOfObjectSortedByValueDescending(selectorAndElementCount); // 按元素选择器对应的元素数量，从多到少排列
        switch (selectorAndElementCountSortByValueDescending.length) {
            case 0:
                break;
            case 1:
                // 只有一个选择器时，则该元素即为"拥有最多对应元素的选择器"
                selectorWithMostElements = selectorAndElementCountSortByValueDescending[0].key;
                break;
            default:
                // 有多个选择器时，则获取第一个选择器，用第一个选择器与第二个选择器作比较，看看第一个选择器对应的元素是否确实比第二个选择器的元素多，防止出现有选择器对应的元素数量相等的情况。
                var firstSelector = selectorAndElementCountSortByValueDescending[0];
                if (firstSelector.value > selectorAndElementCountSortByValueDescending[1].value) {
                    selectorWithMostElements = firstSelector.key;
                }
        }
        if (!selectorWithMostElements) {
            return null;
        }
        var dominantElement = selectorAndElement[selectorWithMostElements];
        return {
            selector: selectorForElement(dominantElement),
            depth: depthOfElementWithinElement(dominantElement, element)
        };
    },

    functionToPreventPruningElementDueToInvisibility: function () {
        var functionToPreventPruning = functionToPreventPruningDueToInvisibilityInWhitelistForHostname(this.contentDocument.location.hostname);

        return functionToPreventPruning || function () {
            return false;
        }
    },

    /**
     * 优化正文、标题、Meta 的节点（清理无用节点，优化结构，加强样式）。
     * @param originalArticleNode 文章节点。
     * @param clonedArticleNode 文章克隆节点。
     * @param cleaningType 清理模式，值从 {@link CleaningType} 中选取。
     * @param allowedToReturnNull 允许返空。
     */
    cleanArticleNode: function (originalArticleNode, clonedArticleNode, cleaningType, allowedToReturnNull) {
        function incrementDepthLevels(delta) {
            depthLevel += delta;
            if (depthInFloat) {
                depthInFloat += delta;
            }
            if (depthInTable) {
                depthInTable += delta;
            }
            if (depthInFontStyle) {
                depthInFontStyle += delta;
            }
            if (depthInFontWeight) {
                depthInFontWeight += delta;
            }
            if (depthInVisibility) {
                depthInVisibility += delta;
            }
        }

        function updateDepthLevelsAfterSiblingTraversal() {
            if (depthInFloat === 1) {
                depthInFloat = 0;
            }
            if (depthInTable === 1) {
                depthInTable = 0;
            }
            if (depthInFontStyle === 1) {
                depthInFontStyle = 0;
            }
            if (depthInFontWeight === 1) {
                depthInFontWeight = 0;
            }
            if (depthInVisibility === 1) {
                depthInVisibility = 0;
            }
        }

        function shouldCleanDueToExcessiveFloatingContent() {
            const MaximumFloatingContentRatio = .8;
            var articleRect = cachedElementBoundingRect(originalArticleNode);

            if (articleRect.width === 0 || articleRect.height === 0) {
                return true;
            }

            var nonFloatingChildren;
            var articleChildren = childrenWithParallelStructure(originalArticleNode);
            var articleChildrenCount = articleChildren.length;

            if (articleChildrenCount) {
                nonFloatingChildren = [];
                for (var i = 0; articleChildrenCount > i; i++) {
                    var child = articleChildren[i];
                    if (getComputedStyle(child)['float'] === 'none') {
                        var grandChildren = child.children;
                        for (var m = 0; m < grandChildren.length; m++) {
                            nonFloatingChildren.push(grandChildren[m]);
                        }
                    } else {
                        nonFloatingChildren.push(child);
                    }
                }
            } else {
                nonFloatingChildren = originalArticleNode.children;
            }

            var nonFloatingChildrenCount = nonFloatingChildren.length;
            var textLength = 0;
            for (var l = 0; l < nonFloatingChildrenCount; ++l) {
                var currentNonFloatingChild = nonFloatingChildren[l];
                if (getComputedStyle(currentNonFloatingChild)["float"] !== 'none') {
                    textLength += currentNonFloatingChild.innerText.length;
                }
            }

            var originalArticleTextLength = originalArticleNode.innerText.length;
            var floatingContentRatio = textLength / originalArticleTextLength;
            return floatingContentRatio > MaximumFloatingContentRatio;
        }

        // TODO: michael 猜测是判断导航
        function o(element) {
            const n = 50;
            if (cachedElementBoundingRect(element).height > n) {
                return false;
            }

            const i = {
                UL: 1,
                LI: 1,
                NAV: 1
            };


            if (i[element.tagName]) {
                return true;
            }

            return element.parentElement === originalArticleNode && !element.nextElementSibling;
        }

        function s(e, t) {
            const n = .9;
            return !(cachedElementBoundingRect(e).height > n * cachedElementBoundingRect(t).height)
        }

        function stylePullQuote(element, fontSizeValue) {
            const pullQuoteWeight = 1.1;
            const fontSizeWeight = 1.4;
            if (fontSizeValue && dominantFontSizeInPoints) {
                var isPulledQuote = fontSizeValue > fontSizeWeight * dominantFontSizeInPoints || pullQuoteRegEx.test(currentElement.className) && fontSizeValue > pullQuoteWeight * dominantFontSizeInPoints;

                if (isPulledQuote && !element.closest('.pullquote')) {
                    element.classList.add('pullquote');
                    if (!element.classList.contains('float')) {
                        element.style.width = null;
                        cleanStyleAndClassList(element);
                    }
                }
            }
        }

        function m(e, t) {
            for (var n = e[t]; n; n = n[t]) {
                if (!isNodeWhitespace(n) && n.nodeType !== Node.COMMENT_NODE) {
                    return false;
                }
            }
            return true;
        }

        const tagNamesToAlwaysPrune = {
            FORM: 1,
            SCRIPT: 1,
            STYLE: 1,
            LINK: 1,
            BUTTON: 1
        };
        const tagNamesToConsiderPruning = {
            DIV: 1,
            TABLE: 1,
            OBJECT: 1,
            UL: 1,
            CANVAS: 1,
            P: 1,
            IFRAME: 1,
            ASIDE: 1,
            SECTION: 1,
            FOOTER: 1,
            NAV: 1,
            OL: 1,
            MENU: 1,
            svg: 1
        };
        const tagNamesAffectingFontStyle = {
            I: 1,
            EM: 1
        };
        const tagNamesAffectingFontWeight = {
            B: 1,
            STRONG: 1,
            H1: 1,
            H2: 1,
            H3: 1,
            H4: 1,
            H5: 1,
            H6: 1
        };
        const LightboxRegEx = /lightbox/i;

        let elementsToConsiderPruning = [];
        let depthLevel = 0,
            depthInFloat = 0,
            depthInTable = 0,
            depthInFontStyle = 0,
            depthInFontWeight = 0,
            depthInVisibility = 0,
            currentElement = originalArticleNode,
            view = currentElement.ownerDocument.defaultView,
            currentCloneElement = clonedArticleNode,
            articleTitle = this.postTitle(),
            articleTitleElement = this._articleTitleElement;

        this.postSubtitle();
        let subheadElement = this._articleSubheadElement;

        let incorrectTitle = articleTitleElement && cachedElementBoundingRect(articleTitleElement).top > cachedElementBoundingRect(originalArticleNode).bottom,
            elementPinToOriginalArticleNode = this.elementPinToEdge(originalArticleNode),
            elementMarker = null, // 用于标记需要自动滚动到的元素位置（如果用户不在页面顶部）
            isArticleVisible = isElementVisible(originalArticleNode),
            headerSet = new Set([articleTitleElement, subheadElement]),
            metaSet = new Set;

        if (cleaningType === CleaningType.MainArticleContent) {
            this.updateArticleBylineAndDateElementsIfNecessary();
            let byLineElement = this.articleBylineElement();

            if (byLineElement) {
                metaSet.add(byLineElement);
            }

            let dateElement = this.articleDateElement();
            if (dateElement) {
                metaSet.add(dateElement);
            }
        }


        var dominantContentSelectorAndDepth = this.dominantContentSelectorAndDepth(originalArticleNode),
            shouldCleanDueToExcessiveFloatingContent = shouldCleanDueToExcessiveFloatingContent(),
            pageURLStrings = new Set;
        this.previouslyDiscoveredPageURLStrings.forEach(function (e) {
            pageURLStrings.add(e)
        });

        // 翻页相关逻辑
        var nextPageURL = this.nextPageURL();
        if (nextPageURL) {
            pageURLStrings.add(nextPageURL);
        }

        var articleTitleRect = null;
        if (this._articleTitleElement) {
            articleTitleRect = cachedElementBoundingRect(this._articleTitleElement);
        }

        var functionToPreventPruningElementDueToInvisibility = this.functionToPreventPruningElementDueToInvisibility(),
            dominantFontAppearance = dominantFontAppearanceForElement(originalArticleNode),
            dominantFontSizeInPoints = dominantFontSizeInPointsFromFontAppearanceString(dominantFontAppearance),
            dominantColor = dominantColorInPointsFromFontAppearanceString(dominantFontAppearance);
        const pullQuoteRegEx = /pull(ed)?quote/i;

        console.log('cleanArticleNode started');


        // 清理文章顶部的无关图文信息，必须在 while 之前完成
        const possibleAdsCoordinates = [
            {x: 35, y: 125},
            {x: 35, y: 150},
            {x: 35, y: 175},
            {x: 35, y: 200}
        ];

        for (let i = 0; i < possibleAdsCoordinates.length; i++) {
            let coordinate = possibleAdsCoordinates[i];
            let element = elementAtPoint(coordinate.x, coordinate.y);
            let elementContainer = element.closest('p');

            // 如果找不到 p，还有可能是直接放在了 section 里面
            if (!elementContainer) {
                elementContainer = element.closest('section');
            }

            // 防止图片没有 data-ratio 属性，就会被去掉，所以这里保护一下
            if (element.tagName === 'IMG' && element.getAttribute('data-ratio') && element.getAttribute('data-ratio') < 0.35) {
                element.setAttribute('shouldCleanAd', 'true');
            }

        }

        var prunedMetaElements = [];
        var Y = [];
        var hadFoundEndOfArticle = false;
        while (currentElement) {
            var prunedElement = null,
                tagName = currentCloneElement.tagName,
                hasFigure = false;

            //  - 判断到需要保留且容易在其他步骤被误判的元素是加入 .protected className 作为保护
            //  - 判断文章上方边界用 articleTitleRect

            // 开始分析不需要的页面元素
            currentCloneElement.originalElement = currentElement;

            if (currentElement === elementPinToOriginalArticleNode) {
                elementMarker = currentCloneElement
            }

            // 移除文章结尾的非正文内容（作者的其他推介，往期回顾等）
            // 这里需要找到合适的文章结尾的标记
            if (currentElement.parentElement === this.article.element) {
                let currentTextContent = currentElement.innerText.replace(/[<>\-~：:—]/g, '').trim();
                let endKeywordsInCurrentElement = currentTextContent.match(KeywordsAboutTheEndOfTheArticle);
                console.log('endKeywordsInCurrentElement', currentTextContent, endKeywordsInCurrentElement);
                if (endKeywordsInCurrentElement) {
                    for (var indexOfEndKeyword = 0; indexOfEndKeyword < endKeywordsInCurrentElement.length; indexOfEndKeyword++) {
                        var keyword = endKeywordsInCurrentElement[indexOfEndKeyword];
                        if (stringSimilarity(currentTextContent, keyword) >= StringSimilarityToDeclareStringsIsTheEndOfTheArticle) {
                            hadFoundEndOfArticle = true;
                        }
                    }
                }
            }

            if (hadFoundEndOfArticle) {
                prunedElement = currentCloneElement;
            }

            if (!depthInVisibility && elementAppearsToBeCollapsed(currentElement)) {
                depthInVisibility = 1;
            }

            // 去掉黑名单 tag
            if (tagName in tagNamesToAlwaysPrune) {
                prunedElement = currentCloneElement;
            }

            // 去掉早前标记的广告
            if (currentElement.getAttribute('shouldCleanAd')) {
                prunedElement = currentCloneElement;
            }

            // 去掉标题（标题是后来贴上的，如果文章正文里找到了标题，则需要去掉）
            if (!prunedElement && currentElement !== originalArticleNode && headerSet.has(currentElement)) {
                prunedElement = currentCloneElement
            } else if (!prunedElement && currentElement !== originalArticleNode && metaSet.has(currentElement)) {
                // meta 信息去掉（因为重复）
                currentCloneElement.parentElementBeforePruning = currentCloneElement.parentElement;
                prunedElement = currentCloneElement;
                prunedMetaElements.push(currentCloneElement);
            } else if (elementIsAHeader(currentCloneElement) && previousLeafElementForElement(currentElement) === articleTitleElement) {
                // 保护文章标题
                currentCloneElement.classList.add("protected");
            }

            // 判断是否去掉 H1 和 H2
            if (!prunedElement && ("H1" === tagName || "H2" === tagName)) {
                var distanceFromOriginalArticleNodeTop = currentElement.offsetTop - originalArticleNode.offsetTop;
                if (distanceFromOriginalArticleNodeTop < HeaderMinimumDistanceFromArticleTop) {
                    var headerText = trimmedInnerTextIgnoringTextTransform(currentElement),
                        maxDistanceToConsiderSimilar = headerText.length * HeaderLevenshteinDistanceToLengthRatio;
                    if (levenshteinDistance(articleTitle, headerText) <= maxDistanceToConsiderSimilar) {
                        prunedElement = currentCloneElement;
                    }
                }
            }

            // Wiki 页去掉编辑按钮
            if (!prunedElement && this.isMediaWikiPage() && /editsection|icon-edit/.test(currentElement.className)) {
                prunedElement = currentCloneElement;
            }

            // 视频元素是否需要保留（取决于 src)
            if (tagName === 'VIDEO') {
                if (currentCloneElement.getAttribute('src')) {
                    currentCloneElement.classList.add('protected');
                    var videoRect = cachedElementBoundingRect(currentElement);
                    currentCloneElement.setAttribute('width', videoRect.width);
                    currentCloneElement.setAttribute('height', videoRect.height);
                    currentCloneElement.setAttribute('controls', true);
                    currentCloneElement.removeAttribute('autoplay');
                    currentCloneElement.removeAttribute('preload');
                    currentCloneElement.removeAttribute('style');
                } else {
                    prunedElement = currentCloneElement;
                }
            }

            var computedStyle;
            if (!prunedElement) {
                computedStyle = getComputedStyle(currentElement);
            }

            // 处理 DIV 的 lazyload
            if (!prunedElement && tagName === 'DIV' && LazyLoadRegex.test(currentElement.className) && !currentElement.innerText) {
                var imageURL = lazyLoadingImageURLForElement(currentCloneElement, currentElement.className);
                if (imageURL) {
                    var imageElementForLazyLoad = this.contentDocument.createElement('img');
                    imageElementForLazyLoad.setAttribute('src', imageURL);
                    currentCloneElement.parentNode.replaceChild(imageElementForLazyLoad, currentCloneElement);
                    currentCloneElement = imageElementForLazyLoad;
                    currentCloneElement.originalElement = currentElement;
                    tagName = currentCloneElement.tagName;
                    prunedElement = currentCloneElement;
                    currentCloneElement.classList.add('protected');
                }
            }

            // 把所有元素替换成 p（一个 node 被 append 到另一个 node 之后自己的 children 会相应减少）
            if (!prunedElement && tagName === 'DIV' && currentCloneElement.parentNode) {
                var elements = currentElement.querySelectorAll('a, blockquote, dl, div, img, ol, p, pre, table, ul');
                var inFloat = depthInFloat || 'none' !== computedStyle['float'];
                if (!inFloat && !elements.length) {
                    var currentElementParentNode = currentCloneElement.parentNode;
                    var replacementNode = this.contentDocument.createElement('p');
                    while (currentCloneElement.firstChild) {
                        var child = currentCloneElement.firstChild;
                        replacementNode.appendChild(child);
                    }
                    currentElementParentNode.replaceChild(replacementNode, currentCloneElement);
                    elementMarker === currentCloneElement && (elementMarker = replacementNode);
                    currentCloneElement = replacementNode;
                    currentCloneElement.originalElement = currentElement;
                    tagName = currentCloneElement.tagName;
                }
            }

            // 如果 tagNames 命中需要考虑去除的，则 push 进对应的数组里
            if (!prunedElement && currentCloneElement.parentNode && tagName in tagNamesToConsiderPruning) {
                elementsToConsiderPruning.push(currentCloneElement);
            }

            if (!prunedElement) {
                // 如果元素为离屏元素，则去掉
                if (isElementPositionedOffScreen(currentElement)) {
                    prunedElement = currentCloneElement
                } else if (currentElement !== originalArticleNode &&
                    !depthInFloat &&
                    'none' !== computedStyle['float'] &&
                    !shouldCleanDueToExcessiveFloatingContent &&
                    (cachedElementBoundingRect(currentElement).height >= FloatMinimumHeight || currentElement.childElementCount > 1)) {
                    // 如果是有效的浮动元素，则做上标记
                    depthInFloat = 1;
                }
            }

            if (!prunedElement) {
                sanitizeElementByRemovingAttributes(currentCloneElement);
                // 处理 meta 信息，替换分隔符
                if (cleaningType === CleaningType.MetadataContent) {
                    if ("|" === currentCloneElement.innerText) {
                        currentCloneElement.innerText = "";
                        currentCloneElement.classList.add("delimiter");
                    } else if ("TIME" === currentCloneElement.tagName) {
                        var previousElementSibling = currentCloneElement.previousElementSibling;
                        if (previousElementSibling && "SPAN" === previousElementSibling.tagName && !previousElementSibling.classList.contains("delimiter")) {
                            var me = this.contentDocument.createElement("span");
                            me.classList.add("delimiter");
                            currentCloneElement.before(me);
                        }
                    } else {
                        "FIGURE" === tagName && (prunedElement = currentCloneElement);
                    }
                }

                // 使用阅读器自带的 clearfix
                if ("both" === computedStyle.clear) {
                    currentCloneElement.classList.add("clear");
                }

                // 判断小程序链接
                if (tagName === 'SPAN' && VeryNegativeClassNameRegEx.test(currentElement.className)) {
                    prunedElement = currentCloneElement;
                }

                // 判断文末推广
                if (tagName === 'SECTION') {
                    const AdSectionDataAttrRegEx = /^Copyright.*playhudong.*All Rights Reserved.$/;

                    if (currentElement.getAttribute('data-mpa-powered-by') === 'yiban.io') {
                        console.log('PSA', '去掉文末推广信息，因为 data attributes 命中');
                        prunedElement = currentCloneElement;
                    }
                }

                // 判断图片说明
                if (tagName === 'SECTION' || tagName === 'P') {
                    var textLength = currentElement.textContent.trim().length;

                    if (textLength > 0 && textLength <= 100) {
                        var previousElement = currentElement.previousElementSibling;
                        var isPreviousElementImage = false;
                        const positiveFigCaptionWordsRegex = /▲|▴|▵|来源|Credit|摄影|图片|撰文|图文|作者图|图为|wikipedia/;
                        const negativeFigCaptionWordsRegex = /相关阅读/;
                        var textContent = currentElement.textContent;
                        var previousElementChildrenImages = previousElement ? previousElement.querySelectorAll('img') : null;

                        if (previousElementChildrenImages && previousElementChildrenImages.length === 1 && !previousElementChildrenImages[0].closest('figure')) {
                            isPreviousElementImage = true;
                        }

                        var figCaptionTextScore = 0;
                        figCaptionTextScore += positiveFigCaptionWordsRegex.exec(textContent) ? positiveFigCaptionWordsRegex.exec(textContent).length : 0;
                        figCaptionTextScore -= negativeFigCaptionWordsRegex.exec(textContent) ? negativeFigCaptionWordsRegex.exec(textContent).length : 0;

                        var figCaptionStyleScore = 0;
                        if (dominantFontSizeFromComputedClosestContainer(currentElement) < dominantFontSizeInPoints) {
                            figCaptionStyleScore += 1;
                        }

                        if (isLighterColor(dominantColor, dominantColorFromComputedClosestContainer(currentElement))) {
                            // 如果本段落文字与文章主要文字相比，颜色较浅，则倾向于本段可能是图片说明文字
                            figCaptionStyleScore += 1;
                        }

                        if (isPreviousElementImage && figCaptionTextScore >= 0 && figCaptionStyleScore >= 1) {
                            currentElement.classList.add('figcaption');
                            currentCloneElement.classList.add('figcaption');
                            console.log('PSA', '命中图片说明规则', currentElement);
                        }
                    }

                }

                // 判断文章段落标题
                if ((tagName === 'SECTION' || tagName === 'P') && !currentElement.classList.contains('figcaption')) {

                    var textLength = currentElement.textContent.trim().length;

                    // 小于等于10个字符的认为可能是标题
                    if (textLength > 0 && textLength <= 20) {

                        var isParagraphHeader = false;
                        const paragraphHeaderRegex = /标题/;
                        var potentialParagraphHeaderAttrs = ['label'];

                        for (var k = 0; k < potentialParagraphHeaderAttrs.length; k++) {
                            if (paragraphHeaderRegex.test(currentElement.getAttribute(potentialParagraphHeaderAttrs[k]))) {
                                isParagraphHeader = true;
                                break;
                            }
                        }

                        if (!isParagraphHeader) {
                            var previousParagraph = currentElement.previousElementSibling,
                                nextParagraph = currentElement.nextElementSibling,
                                previousParagraphLength = previousParagraph ? previousParagraph.textContent.trim().length : 0,
                                nextParagraphLength = nextParagraph ? nextParagraph.textContent.trim().length : 0;

                            var scoreWhenParagraphIsHeader = 3,
                                currentScore = 0;

                            // 如果前后段落（如果有）的字数比本段长，则本段可能是标题
                            if (previousParagraphLength > 0 && previousParagraphLength > textLength) {
                                currentScore += 1;
                            }

                            if (nextParagraphLength > 0 && nextParagraphLength > textLength) {
                                currentScore += 1;
                            }

                            // 根据样式判断本段是否为标题
                            if (dominantTextIsBold(currentElement)) {
                                currentScore += 2;
                            }

                            if (dominantFontSizeFromComputedClosestContainer(currentElement) > dominantFontSizeInPoints) {
                                currentScore += 2;
                            }

                            if (dominantFontSizeFromComputedClosestContainer(currentElement) < dominantFontSizeInPoints) {
                                currentScore -= 2;
                            }

                            var dominantStyleOfCurrentElementFromComputedClosestContainer = dominantStyleFromComputedClosestContainer(currentElement);
                            if (dominantStyleOfCurrentElementFromComputedClosestContainer.color !== dominantColor) {
                                currentScore += 2;
                            }

                            var dominantHSLColorOfCurrentElementFromComputedClosestContainer = formatColorToHsl(dominantStyleOfCurrentElementFromComputedClosestContainer.color);
                            var dominantHSLColor = formatColorToHsl(dominantColor);
                            let dominantBackgroundColorOfCurrentElementFromComputedClosestContainer = dominantStyleOfCurrentElementFromComputedClosestContainer.backgroundColor;
                            if (isLighterColor(dominantHSLColorOfCurrentElementFromComputedClosestContainer, dominantHSLColor, true) && dominantBackgroundColorOfCurrentElementFromComputedClosestContainer === 'rgba(0, 0, 0, 0)') {
                                currentScore -= 2;
                            }

                            if (currentScore >= scoreWhenParagraphIsHeader) {
                                isParagraphHeader = true;
                            }
                        }

                        if (isParagraphHeader) {
                            var headerParentNode = currentCloneElement.parentNode;
                            var replacementHeader = this.contentDocument.createElement('h3');
                            while (currentCloneElement.firstChild) {
                                var headerChild = currentCloneElement.firstChild;
                                replacementHeader.appendChild(headerChild);
                            }
                            headerParentNode.replaceChild(replacementHeader, currentCloneElement);
                            elementMarker === currentCloneElement && (elementMarker = replacementHeader);
                            currentCloneElement = replacementHeader;
                            currentCloneElement.originalElement = currentElement;
                            tagName = currentCloneElement.tagName;

                            console.log('PSA', '命中了段落标题', currentCloneElement, replacementHeader);
                        }
                    }
                }

                // 处理列表，判断是否需要保留项目符合（包括原生的和自己实现的）
                if ("UL" === tagName || "OL" === tagName || "MENU" === tagName) {
                    if (articleTitleRect && cachedElementBoundingRect(currentElement).top < articleTitleRect.top) {
                        prunedElement = currentCloneElement;
                    } else if ("none" === computedStyle["list-style-type"] && "none" === computedStyle["background-image"]) {
                        var children = currentElement.children;
                        var childrenCount = children.length;
                        var ignoreBulletPoint = true;
                        for (var ge = 0; childrenCount > ge; ++ge) {
                            var currentChild = children[ge],
                                currentChildStyle = getComputedStyle(currentChild);
                            if ("none" !== currentChildStyle["list-style-type"] || 0 !== parseInt(currentChildStyle["-webkit-padding-start"])) {
                                ignoreBulletPoint = false;
                                break
                            }
                            var ve = getComputedStyle(currentChild, ":before").content;
                            const knownBulletPointRegEx = /\u2022|\u25e6|\u2023|\u2219|counter/;
                            if (knownBulletPointRegEx.test(ve)) {
                                ignoreBulletPoint = false;
                                break;
                            }
                        }

                        if (ignoreBulletPoint) {
                            currentCloneElement.classList.add("list-style-type-none");
                        }
                    }

                    // 处理代码块
                    if (currentElement.querySelector("code")) {
                        const monospaceRegEx = /monospace|menlo|courier/i;
                        var dominantFontFamilyAndSize = dominantFontAppearanceForElement(currentElement);

                        if (monospaceRegEx.test(dominantFontFamilyAndSize)) {
                            currentCloneElement.classList.add("code-block");
                            currentCloneElement.classList.add("protected");
                        }
                    }
                }

                // 处理 fontStyle
                if (!depthInFontStyle && computedStyle.fontStyle !== "normal") {
                    if (!(tagName in tagNamesAffectingFontStyle))
                        currentCloneElement.style.fontStyle = computedStyle.fontStyle;
                    depthInFontStyle = 1;
                }

                // 处理 fontWeight
                if (!depthInFontWeight && computedStyle.fontWeight !== "normal") {
                    if (!(tagName in tagNamesAffectingFontWeight)) {
                        currentCloneElement.style.fontWeight = computedStyle.fontWeight;

                        var fontWeightInt = parseInt(computedStyle.fontWeight),
                            fontWeightValue = null;

                        if (isNaN(fontWeightInt)) {
                            fontWeightValue = computedStyle.fontWeight;
                        } else {
                            // 有部分字体提供 400 到 500 之间到特殊字重，由于这些字重还认为是普通，所有不作处理；除此之外粗细字体均做加粗处理
                            400 >= fontWeightInt || fontWeightInt >= 500 && (fontWeightValue = "bold");
                        }

                        if (fontWeightValue) {
                            currentCloneElement.style.fontWeight = fontWeightValue;
                        }
                    }
                    depthInFontWeight = 1;
                }

                // 处理悬浮 section
                if (depthInFloat && "SECTION" !== tagName && s(currentElement, originalArticleNode) || "ASIDE" === tagName) {
                    var fontSizeString = dominantFontAppearanceForElement(currentElement),
                        fontSizeValue = dominantFontSizeInPointsFromFontAppearanceString(fontSizeString),
                        fontSizeEqualsToDominantText = fontSizeString && fontSizeString === dominantFontAppearance;
                    if (1 === depthInFloat &&
                        (cachedElementBoundingRect(currentElement).width <= MaximumFloatWidth ?
                            currentCloneElement.setAttribute("class", "auxiliary float " + computedStyle["float"]) :
                            fontSizeEqualsToDominantText || currentCloneElement.classList.add("auxiliary")),
                            currentCloneElement.closest(".auxiliary")) {
                        var widthValue = currentElement.style.getPropertyValue("width");
                        if ("table" === computedStyle.display && /%/.test(widthValue) && parseInt(widthValue) < 2)
                            currentCloneElement.style.width = computedStyle.width;
                        else if (widthValue)
                            currentCloneElement.style.width = widthValue;
                        else {
                            // 取得元素的 CSS 规则
                            var rules = view.getMatchedCSSRules(currentElement, "", true);
                            if (rules)
                                for (var length = rules.length, i = length - 1; i >= 0; --i) {
                                    widthValue = rules[i].style.getPropertyValue("width");
                                    var widthInt = parseInt(widthValue);
                                    if (widthValue && (isNaN(widthInt) || widthInt > 0)) {
                                        currentCloneElement.style.width = widthValue;
                                        break;
                                    }
                                }
                        }
                        if (depthInFloat === 1 && !widthValue) {
                            currentCloneElement.style.width = cachedElementBoundingRect(currentElement).width + "px";
                        }
                    }

                    stylePullQuote(currentCloneElement, fontSizeValue);
                }

                if ("TABLE" === tagName) {
                    if (!depthInTable) {
                        depthInTable = 1;
                    }
                } else if ("IMG" === tagName) {
                    var imageURL = lazyLoadingImageURLForElement(currentCloneElement, currentElement.className);
                    if (imageURL) {
                        currentCloneElement.setAttribute("src", imageURL);
                        var hasFigureContainer = !!currentCloneElement.closest("figure");
                        if (!hasFigureContainer) {
                            var currentElementAttrs = currentElement.attributes;
                            var currentElementAttrsCount = currentElementAttrs.length;
                            for (var i = 0; currentElementAttrsCount > i; ++i)
                                if (LightboxRegEx.test(currentElementAttrs[i].nodeName)) {
                                    hasFigureContainer = true;
                                    break;
                                }
                        }

                        if (hasFigureContainer) {
                            currentCloneElement.classList.add("protected");
                            hasFigure = true;
                        }
                    }

                    // 干掉老式 inline 样式
                    currentCloneElement.removeAttribute("border");
                    currentCloneElement.removeAttribute("hspace");
                    currentCloneElement.removeAttribute("vspace");
                    var currentElementAlignment = currentCloneElement.getAttribute("align");
                    currentCloneElement.removeAttribute("align");

                    // 判断是不是 float
                    if ("left" === currentElementAlignment || "right" === currentElementAlignment) {
                        currentCloneElement.classList.add("float");
                        currentCloneElement.classList.add(currentElementAlignment);
                    }

                    if (!depthInFloat && !hasFigure) {
                        var imageBoundingRect = cachedElementBoundingRect(currentElement),
                            imageWidth = imageBoundingRect.width,
                            imageHeight = imageBoundingRect.height;

                        if (imageIsContainedByContainerWithImageAsBackgroundImage(currentElement)) {
                            currentCloneElement.classList.add("protected")
                        } else if (1 === imageWidth && 1 === imageHeight) {
                            // 如果是 1x1 的图片则去掉
                            prunedElement = currentCloneElement;
                        } else if (articleTitleRect && imageHeight < MinimumHeightForImagesAboveTheArticleTitle && imageBoundingRect.bottom < articleTitleRect.top) {
                            // 说明是题图，去掉
                            prunedElement = currentCloneElement;
                        } else {
                            // 如果图片太小，则加上小图的 class（例如QQ表情）
                            if (imageWidth < ImageSizeTiny && imageHeight < ImageSizeTiny) {
                                currentCloneElement.setAttribute("class", "reader-image-tiny");
                            }
                        }
                    }

                    // 清理 meta 的时候，如果里面有太大的元素，则去掉那些过大的元素
                    if (cleaningType === CleaningType.MetadataContent) {
                        var elementRect = cachedElementBoundingRect(currentElement);
                        if (elementRect.width > MaximumWidthOrHeightOfImageInMetadataSection || elementRect.height > MaximumWidthOrHeightOfImageInMetadataSection) {
                            prunedElement = currentCloneElement;
                        }
                    }

                } else if ("FONT" === tagName) {
                    currentCloneElement.removeAttribute("size");
                    currentCloneElement.removeAttribute("face");
                    currentCloneElement.removeAttribute("color");
                } else if ("A" === tagName && currentCloneElement.parentNode) {
                    // 处理 A

                    var href = currentCloneElement.getAttribute("href");

                    if ("author" === currentElement.getAttribute("itemprop")) {
                        currentCloneElement.classList.add("protected");
                    } else if (href && href.length && ("#" === href[0] || anchorRunsJavaScriptOnActivation(currentCloneElement))) {
                        // 脚注
                        const we = {
                            LI: 1,
                            SUP: 1
                        };
                        if (!depthInTable && !currentCloneElement.childElementCount && currentCloneElement.parentElement.childElementCount === 1 && !we[currentCloneElement.parentElement.tagName]) {
                            var qe = this.contentDocument.evaluate("text()", currentCloneElement.parentElement, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null);
                            qe.snapshotLength || (prunedElement = currentCloneElement)
                        }
                        if (!prunedElement) {
                            var replacementNode = this.contentDocument.createElement("span");
                            if (1 === currentCloneElement.childElementCount && "IMG" === currentCloneElement.firstElementChild.tagName) {
                                var imageElement = currentCloneElement.firstElementChild;
                                if (imageElement.width > AnchorImageMinimumWidth && imageElement.height > AnchorImageMinimumHeight) {
                                    replacementNode.setAttribute("class", "converted-image-anchor");
                                }
                            }
                            for (replacementNode.className || replacementNode.setAttribute("class", "converted-anchor"); currentCloneElement.firstChild;)
                                replacementNode.appendChild(currentCloneElement.firstChild);
                            currentCloneElement.parentNode.replaceChild(replacementNode, currentCloneElement);
                            currentCloneElement = replacementNode;
                            elementMarker === currentCloneElement && (elementMarker = replacementNode)
                        }
                    } else if (AdvertisementHostRegex.test(currentCloneElement.host) && !currentCloneElement.innerText) {
                        prunedElement = currentCloneElement;
                    } else if (articleTitleElement && !incorrectTitle && articleTitleElement.compareDocumentPosition(currentElement) & document.DOCUMENT_POSITION_PRECEDING && cachedElementBoundingRect(currentElement).top < cachedElementBoundingRect(articleTitleElement).top) {
                        Y.push(currentCloneElement);
                    } else {
                        var We = currentElement.children;
                        1 === We.length && "IMG" === We[0].tagName && !currentElement.innerText && anchorLooksLikeDownloadFlashLink(currentElement) && (prunedElement = currentCloneElement)
                    }
                } else if ("ASIDE" === tagName || "BLOCKQUOTE" === tagName || "Q" === tagName || "DIV" === tagName && pullQuoteRegEx.test(currentElement.className)) {
                    var fontSizeString = dominantFontAppearanceForElement(currentElement),
                        fontSizeValue = dominantFontSizeInPointsFromFontAppearanceString(fontSizeString);
                    stylePullQuote(currentCloneElement, fontSizeValue)
                }

            }
            if (computedStyle && isArticleVisible && !hasFigure) {
                var Ue = "none" === computedStyle.display || "visible" !== computedStyle.visibility || computedStyleIndicatesElementIsInvisibleDueToClipping(computedStyle);
                if (Ue && !depthInVisibility) {
                    var He = dominantContentSelectorAndDepth ? depthLevel === dominantContentSelectorAndDepth.depth && selectorForElement(currentElement) === dominantContentSelectorAndDepth.selector : false;
                    He || functionToPreventPruningElementDueToInvisibility(currentElement, originalArticleNode) || (prunedElement = currentCloneElement)
                }
            }

            if (!prunedElement && elementIsCommentBlock(currentElement)) {
                prunedElement = currentCloneElement;
            }

            if (!prunedElement && articleTitleRect && cachedElementBoundingRect(currentElement).top < articleTitleRect.top && VeryLiberalCommentRegex.test(currentElement.className) && currentCloneElement.parentElement) {
                prunedElement = currentCloneElement;
            }

            if (!prunedElement && "A" === tagName && pageURLStrings.has(currentElement.href)) {
                for (let Ve, ze, Ge = currentElement, je = currentCloneElement;
                     (Ge = Ge.parentElement) && (je = je.parentElement);) {
                    const Ye = 10;
                    if (cachedElementBoundingRect(Ge).top - cachedElementBoundingRect(currentElement).top > Ye)
                        break;
                    if (Ge === originalArticleNode)
                        break;

                    if (o(Ge)) {
                        Ve = Ge;
                        ze = je;
                    }
                }

                if (Ve) {
                    prunedElement = ze;
                    currentElement = Ve;
                    currentCloneElement = ze;
                    currentCloneElement.originalElement = currentElement;
                    tagName = currentCloneElement.tagName;
                }

                Ge = null;
                je = null;
                Ve = null;
                ze = null;
            }

            !prunedElement || prunedElement.parentElement || allowedToReturnNull || (prunedElement = null);

            var Xe = prunedElement ? null : currentElement.firstElementChild;

            if (Xe) {
                currentElement = Xe;
                currentCloneElement = currentCloneElement.firstElementChild;
                incrementDepthLevels(1);
            } else {
                for (var Ke; currentElement !== originalArticleNode && !(Ke = currentElement.nextElementSibling);) {
                    currentElement = currentElement.parentElement;
                    currentCloneElement = currentCloneElement.parentElement;
                    incrementDepthLevels(-1);
                }

                if (currentElement === originalArticleNode) {
                    if (prunedElement && !elementIsProtected(prunedElement)) {
                        if (prunedElement.parentElement) {
                            console.log('prunedElement 去掉了一个元素', prunedElement);
                            prunedElement.remove();
                        } else if (allowedToReturnNull) {
                            return null;
                        }
                    }
                    break;
                }

                currentElement = Ke;
                currentCloneElement = currentCloneElement.nextElementSibling;
                updateDepthLevelsAfterSiblingTraversal();
            }

            if (prunedElement && !elementIsProtected(prunedElement)) {
                if (prunedElement.parentElement) {
                    console.log('prunedElement 去掉了一个元素 2', prunedElement);
                    prunedElement.remove();
                } else if (allowedToReturnNull) {
                    return null;
                }
            }
        }

        // 如果前面有找到明确的文章结尾，则把结尾以及后续的内容都去掉
        var elementsHaveEndSymbol = clonedArticleNode.querySelectorAll('.endSymbol');
        if (elementsHaveEndSymbol.length > 0) {
            var elementAfterEnd = [];
            var elementContainsTheEndOfTheArticle = elementsHaveEndSymbol.pop();
            while (elementContainsTheEndOfTheArticle) {
                elementAfterEnd.push(elementContainsTheEndOfTheArticle);
                elementContainsTheEndOfTheArticle = elementContainsTheEndOfTheArticle.nextElementSibling;
            }

            for (var i = 0; i < elementAfterEnd.length; i++) {
                elementAfterEnd[i].remove();
            }
        }

        // 处理公众号中的内嵌 QQ 音乐和语音
        let MPAudioElements = clonedArticleNode.querySelectorAll('mpvoice, qqmusic');
        if (MPAudioElements && MPAudioElements.length > 0) {
            for (let i = 0; i < MPAudioElements.length; i++) {
                let MPAudioElement = MPAudioElements[i];
                let contentURL;
                if (MPAudioElement.tagName === 'MPVOICE') {
                    // 语音
                    let mediaId = MPAudioElement.getAttribute('voice_encode_fileid');
                    contentURL = 'https://res.wx.qq.com/voice/getvoice?mediaid=' + mediaId;
                } else if (MPAudioElement.tagName === 'QQMUSIC') {
                    // QQ 音乐
                    contentURL = MPAudioElement.getAttribute('audiourl');
                }
                if (contentURL) {
                    let nextElementOfMPAudioElement = MPAudioElement.nextElementSibling;
                    let audioParentNode = MPAudioElement.parentNode;
                    let audioData = {contentURL: contentURL};
                    let audioPlayer = audioPlayerWithAudioData(audioData, this.contentDocument);
                    audioParentNode.appendChild(audioPlayer);
                    if (nextElementOfMPAudioElement != null) {
                        nextElementOfMPAudioElement.remove();
                    }
                    console.log('格式化了音频', MPAudioElement);
                }
            }
        }

        // 处理 iFrame
        var iFrameArray = clonedArticleNode.querySelectorAll("iframe");
        var iFrameArrayCount = iFrameArray.length;
        for (var i = 0; iFrameArrayCount > i; ++i) {
            var currentIFrame = iFrameArray[i];
            // 如果是内嵌 Twitter
            if (elementLooksLikeEmbeddedTweet(currentIFrame.originalElement)) {
                // 替换为简化版的内嵌 Twitter
                var cookedSimpleTweetIframe = this.cookedSimpleTweetFromTwitterIframe(currentIFrame);
                if (cookedSimpleTweetIframe) {
                    currentIFrame.parentElement.replaceChild(cookedSimpleTweetIframe, currentIFrame);
                }
            }
            // 增加 allow-scripts allow-same-origin 的沙箱，支持该内容的 JS 交互等（主要是为了支持 Twitter 的交互，如转推等）
            currentIFrame.classList.add("protected");
            currentIFrame.setAttribute("sandbox", "allow-scripts allow-same-origin");
        }

        // 把剩下有可能被出去的元素传入 shouldPruneElement 做进一步判断
        for (var i = elementsToConsiderPruning.length - 1; i >= 0; i--) {
            var element = elementsToConsiderPruning[i];

            if (element.parentNode && this.shouldPruneElement(element, element.originalElement, dominantFontAppearance)) {
                if (elementMarker === element) {
                    (elementMarker = element.nextElementSibling) || (elementMarker = element.parentElement);
                }

                console.log('cleanArticleNode 去掉了一个元素 - shouldPruneElement 返回 true', element);
                element.remove();
            }
        }

        for (var ge = 0; ge < Y.length; ge++) {
            console.log('cleanArticleNode 去掉了一个元素 2', Y[ge]);
            Y[ge].remove();
        }

        var floatElements = clonedArticleNode.querySelectorAll(".float");
        for (var i = 0; i < floatElements.length; ++i) {
            var pruneFloatedElement = false;
            var floatElement = floatElements[i];
            if (!pruneFloatedElement) {
                var anchors = floatElement.querySelectorAll("a, span.converted-image-anchor");
                var replacedAnchors = floatElement.querySelectorAll("span.converted-anchor");
                pruneFloatedElement = floatElement.parentNode && replacedAnchors.length > anchors.length;
            }
            if (!pruneFloatedElement) {
                var plugInsInClonedElement = floatElement.querySelectorAll("embed, object").length,
                    plugInsInOriginalElement = floatElement.originalElement.querySelectorAll("embed, object").length;
                if (!plugInsInClonedElement && plugInsInOriginalElement) {
                    pruneFloatedElement = true;
                }
            }
            if (!pruneFloatedElement) {
                var imagesInOriginalElement = floatElement.originalElement.getElementsByTagName("img");
                var visibleImagesInOriginalElementCount = 0;
                for (var j = 0; j < imagesInOriginalElement.length; j++) {
                    if (isElementVisible(imagesInOriginalElement[j]))
                        visibleImagesInOriginalElementCount++;
                    if (visibleImagesInOriginalElementCount > 1)
                        break;
                }

                if (visibleImagesInOriginalElementCount === 1) {
                    var imagesInClonedElementCount = floatElement.getElementsByTagName("img").length;
                    if (!imagesInClonedElementCount)
                        pruneFloatedElement = true;
                }
            }

            if (!pruneFloatedElement) {
                const gt = "img, video, embed, iframe, object, svg";

                // 空白元素，且不满足上面这些预期的空白元素，则需要去除
                if (!/\S/.test(floatElement.innerText) && !floatElement.matches(gt) && !floatElement.querySelector(gt)) {
                    pruneFloatedElement = true;
                }
            }

            if (pruneFloatedElement) {
                if (elementMarker === floatElement) {
                    elementMarker = floatElement.nextElementSibling ? floatElement.nextElementSibling : floatElement.parentElement;
                }

                if (!elementIsProtected(floatElement)) {
                    console.log('cleanArticleNode 去掉了一个元素 3', floatElement);
                    floatElement.remove();
                }
            }
        }
        for (var ft = clonedArticleNode.querySelectorAll("br"), pt = ft.length, ge = pt - 1; ge >= 0; --ge) {
            var vt = ft[ge];
            vt.originalElement && "block" === getComputedStyle(vt.originalElement.parentElement).display && (m(vt, "nextSibling") || m(vt, "previousSibling")) && vt.remove()
        }
        if (allowedToReturnNull && !removeWhitespace(clonedArticleNode.innerText).length && cleaningType !== CleaningType.LeadingFigure)
            return null;
        if (elementMarker) {
            var Et = document.createElement("div"),
                Nt = elementMarker.originalElement.getBoundingClientRect(),
                St = Nt.height > 0 ? 100 * Nt.top / Nt.height : 0;
            Et.style.position = "relative", Et.style.top = Math.round(-St) + "%", Et.setAttribute("id", "safari-reader-element-marker"), elementMarker.insertBefore(Et, elementMarker.firstChild)
        }
        for (var Tt = {}, at = clonedArticleNode.querySelectorAll("a"), yt = at.length, ge = 0; yt > ge; ++ge) {
            var At = at[ge],
                bt = At.style.fontWeight;
            Tt[bt] || (Tt[bt] = []), Tt[bt].push(At)
        }
        for (var bt in Tt) {
            var Ct = Tt[bt],
                xt = Ct.length;
            if (xt === yt)
                for (var ge = 0; xt > ge; ++ge) {
                    var At = Ct[ge];
                    At.style.fontWeight = null, "" === At.getAttribute("style") && (At.style = null)
                }
        }

        // 处理潜在的广告图片
        var imageElements = clonedArticleNode.querySelectorAll("img");
        var imageElementCount = imageElements.length;

        for (var i = 0; i < imageElementCount; i++) {
            var imageElement = imageElements[i];

            // 广告图片
            if (imageAppearsToBeAd(imageElement)) {
                console.log('PSA', '干掉了一个图片，命中广告', imageElement);
                imageElement.remove();
            }

            // 部分公众号编辑器会将图片的文件名保留为 title，部分二维码的 title 中包含了文件名 qrcode
            if (QRCodeTitleRegEx.test(imageElement.getAttribute('title'))) {
                console.log('PSA', '干掉一个二维码，title 命中');
                imageElement.remove();
            }

            // 如果在可疑图片（比例为1:1）所在 p 前后的 p 中出现了关于公众号关注相关的引导，则把图片干掉
            if (parseInt(imageElement.getAttribute('data-ratio')) === 1) {

                const possibleImageContainer = ['p'];
                var qrCodeContainer = null;

                for (var j = 0; j < possibleImageContainer.length; j++) {
                    if (!qrCodeContainer) {
                        qrCodeContainer = imageElement.closest(possibleImageContainer[j]);
                    }
                }

                if (qrCodeContainer) {

                    let previousParagraph = qrCodeContainer.previousSibling;
                    let nextParagraph = qrCodeContainer.nextSibling;

                    while (previousParagraph && previousParagraph.previousSibling && previousParagraph.innerText.length === 0) {
                        previousParagraph = previousParagraph.previousSibling;
                    }

                    while (nextParagraph && nextParagraph.nextSibling && nextParagraph.innerText.length === 0) {
                        nextParagraph = nextParagraph.nextSibling;
                    }

                    let shouldRemoveImage = false;

                    // 前面段落有广告词汇
                    if (previousParagraph && QRCodeInstructionRegEx.test(previousParagraph.innerText)) {
                        console.log('PSA', '干掉图片之前的段落，关键词命中', previousParagraph.innerText);
                        previousParagraph.remove();
                        shouldRemoveImage = true;
                    }

                    // 后面段落有广告词汇
                    if (nextParagraph && QRCodeInstructionRegEx.test(nextParagraph.innerText)) {
                        console.log('PSA', '干掉图片之后的段落，关键词命中', nextParagraph.innerText);
                        nextParagraph.remove();
                        shouldRemoveImage = true;
                    }

                    // 如果前后文中有一个命中，则图片也要去掉
                    if (shouldRemoveImage) {
                        console.log('PSA', '因为前后文有命中，干掉一个二维码');
                        imageElement.remove();
                    }
                }
            }
        }

        // 清理标记
        for (var Dt = clonedArticleNode.querySelectorAll(".protected"), It = Dt.length, ge = 0; It > ge; ++ge) {
            var et = Dt[ge];
            et.classList.remove('protected');
            et.removeAttribute('shouldCleanAd');
            if (!et.classList.length) {
                et.removeAttribute('class');
            }
        }
        for (var Rt = clonedArticleNode.querySelectorAll("p.auxiliary"), Lt = Rt.length, ge = 0; Lt > ge; ++ge) {
            for (var Mt = Rt[ge], Bt = [Mt], Ft = Mt.nextElementSibling; Ft && "P" === Ft.tagName && Ft.classList.contains("auxiliary");)
                Bt.push(Ft), Ft = Ft.nextElementSibling;
            var Ot = Bt.length;
            if (Ot > 1) {
                for (var ht = 0; Ot > ht; ++ht) {
                    var Pt = Bt[ht];
                    Pt.classList.remove("auxiliary"), Pt.style.width = null, cleanStyleAndClassList(Pt)
                }
                ge += Ot - 1
            }
        }


        for (var _t = prunedMetaElements.length, ge = 0; _t > ge; ++ge) {
            var wt = prunedMetaElements[ge],
                qt = wt.parentElementBeforePruning,
                kt = null,
                Wt = null;
            if (qt)
                var kt = depthOfElementWithinElement(qt, clonedArticleNode),
                    Wt = selectorForElement(qt);
            var listWrapper = qt ? qt.closest("ul") : null;
            if (listWrapper)
                listWrapper.remove();
            else {
                const Ht = 40;
                qt && cachedElementBoundingRect(qt.originalElement).height < Ht && (!dominantContentSelectorAndDepth || dominantContentSelectorAndDepth.selector !== Wt || dominantContentSelectorAndDepth.depth !== kt) ? qt.remove() : wt.remove()
            }
        }
        return clonedArticleNode;
    },
    cookedSimpleTweetFromTwitterIframe: function (e) {
        var t = function (e) {
                var t = this.contentDocument.createElement("div"),
                    n = this.contentDocument.createTextNode(e);
                return t.appendChild(n), t.innerHTML
            }.bind(this),
            n = e.originalElement.contentDocument.documentElement,
            i = n.querySelector("[data-tweet-id].expanded");
        if (!i)
            return null;
        var r = this.contentDocument.createElement("div");
        r.classList.add("tweet-wrapper");
        var a = this.contentDocument.createElement("blockquote");
        a.classList.add("simple-tweet"), r.appendChild(a);
        var l = i.getAttribute("data-tweet-id");
        r.setAttribute("data-reader-tweet-id", l);
        var o = i.querySelector(".dateline"),
            s = i.querySelector('[data-scribe="element:screen_name"]'),
            c = i.querySelector('[data-scribe="element:name"]'),
            m = i.querySelector(".e-entry-title");
        if (!(o && s && c && m))
            return r;
        var d = "&mdash; " + t(c.innerText) + " (" + t(s.innerText) + ")",
            h = this.contentDocument.createElement("p");
        h.innerHTML = m.innerHTML, a.appendChild(h), a.insertAdjacentHTML("beforeend", d);
        var u = this.contentDocument.createElement("span");
        u.innerHTML = o.innerHTML, a.appendChild(u);
        for (var g = a.querySelectorAll("img.twitter-emoji"), f = g.length, p = 0; f > p; ++p) {
            var v = g[p],
                E = v.getAttribute("alt");
            if (E && E.length > 0) {
                var N = this.contentDocument.createElement("span");
                N.innerText = E, v.parentNode.replaceChild(N, v)
            }
        }
        for (var S = a.getElementsByTagName("*"), T = S.length, p = 0; T > p; ++p) {
            var y = S[p];
            "SCRIPT" === y.tagName ? y.remove() : sanitizeElementByRemovingAttributes(y)
        }
        return r
    },
    leadingImageNode: function () {
        const e = 250,
            t = .5,
            n = .9,
            i = 3;
        if (!this.article || !this.article.element)
            return null;
        for (var r = this.article.element, a = 0; i > a && r.parentNode; ++a) {
            r = r.parentNode;
            var l = r.getElementsByTagName("img")[0];
            if (l && isElementVisible(l)) {
                var o = cachedElementBoundingRect(l),
                    s = o.width >= window.innerWidth * n;
                if (!s && o.height < e)
                    continue;
                if (o.width < this._articleWidth * t)
                    continue;
                var c = this.article.element.compareDocumentPosition(l);
                if (!(c & Node.DOCUMENT_POSITION_PRECEDING) || c & Node.DOCUMENT_POSITION_CONTAINED_BY)
                    continue;
                if (c = this.extraArticle ? this.extraArticle.element.compareDocumentPosition(l) : null, c && (!(c & Node.DOCUMENT_POSITION_PRECEDING) || c & Node.DOCUMENT_POSITION_CONTAINED_BY))
                    continue;
                return l
            }
        }
        return null
    },
    pageImageURLFromMetadata: function () {
        var e = this.contentDocument,
            t = e.querySelector("meta[property='og:image']");
        return t || (t = e.querySelector("meta[property='twitter:image']")), t || (t = e.querySelector("meta[property='twitter:image:src']")), t && t.content ? t.content : null
    },
    // 页面的题图片
    mainImageNode: function () {
        var e = this.leadingImageNode();
        if (e)
            return e;
        if (this.article && this.article.element)
            for (var t = this.article.element.querySelectorAll("img"), n = t.length, i = 0; n > i; ++i) {
                var r = t[i],
                    a = r._cachedElementBoundingRect;
                if (a || (a = r.getBoundingClientRect()), a.width >= MainImageMinimumWidthAndHeight && a.height >= MainImageMinimumWidthAndHeight)
                    return r
            }
        return null
    },
    schemaDotOrgMetadataObject: function () {
        if (this._schemaDotOrgMetadataObject)
            return this._schemaDotOrgMetadataObject;
        var e = this.contentDocument.querySelectorAll("script[type='application/ld+json']"),
            t = e.length;
        try {
            for (var n = 0; t > n; ++n) {
                var i = e[n],
                    r = JSON.parse(i.textContent),
                    a = r["@context"];
                if ("https://schema.org" === a || "http://schema.org" === a)
                    return this._schemaDotOrgMetadataObject = r, r
            }
            return null
        } catch (l) {
            return null
        }
    },

    /**
     * 寻找文章标题，根据下面三个步骤，逐一寻找文章标题节点。
     * 根据 DOM 结构找。
     * 根据 <title /> 找。
     * 根据 URL 内容，通过莱文斯坦距离从文章正文内容中筛选。
     */

    postTitle: function () {
        function isPrefixOrSuffix(headerText, documentTitle) {
            var position = headerText ? documentTitle.indexOf(headerText) : -1;
            return (position != -1 && (position == 0 || position + headerText.length == documentTitle.length));
        }

        function t(e, t) {
            return e.host === t.host && e.pathname === t.pathname && e.hash === t.hash
        }

        if (this.articleNode()) {
            if (this._articleTitle)
                return this._articleTitle;
            const HeaderMaximumDistance = 500;
            const i = 20;
            const HeaderMinimumTextLength = 8;
            const HeaderFontSizeBonusMinimumRatio = 1.1;
            const HeaderFontSizeBonusMultiplier = 1.25;
            const HeaderBonusRegEx = /header|title|headline|instapaper_title/i,
                s = 1.5,
                c = 1.8,
                m = 1.5,
                d = .6,
                h = 3,
                u = 1.5,
                g = .8,
                f = 9,
                p = 1.5,
                ByLineBonusRegEx = /byline|author|post-date|rich_media_meta_list|artinfo/i;
            var searchHeadlineBySelector = function (e, headlineSelector) {
                var n = this.contentFromUniqueMetadataSelector(e, headlineSelector);
                if (n) {
                    var i = this.articleTitleAndSiteNameFromTitleString(n);
                    if (i) {
                        n = i.postTitle;
                    }
                }
                return n
            }.bind(this);
            var N = function () {
                    for (var e = this.articleNode(); e; e = e.parentElement)
                        if (elementIndicatesItIsASchemaDotOrgArticleContainer(e))
                            return e;
                    return null
                }.bind(this)(),
                S = N ? this.contentFromUniqueMetadataSelector(N, "meta[itemprop=headline]") : "",
                T = N ? this.contentFromUniqueMetadataSelector(N, "meta[itemprop=alternativeHeadline]") : "",
                y = this.contentDocument,
                A = y.title,
                b = searchHeadlineBySelector(y, "head meta[property='og:title']"),
                C = this.contentFromUniqueMetadataSelector(y, "head meta[property='og:site_name']"),
                x = searchHeadlineBySelector(y, "head meta[name='twitter:title']"),
                sailthruHeadline = searchHeadlineBySelector(y, "meta[name='sailthru.headline']"), // Sailthru 是个用于 SEO 的软件
                I = this.schemaDotOrgMetadataObject(),
                R = I ? I.headline : null,
                L = cachedElementBoundingRect(this.articleNode());
            this.extraArticleNode() && this.extraArticle.isPrepended && (L = cachedElementBoundingRect(this.extraArticleNode()));
            var M = L.left + L.width / 2,
                B = L.top,
                F = B;
            if (this._articleWidth = L.width, this.leadingImage = this.leadingImageNode(), this.leadingImage) {
                var O = cachedElementBoundingRect(this.leadingImage);
                F = (O.top + B) / 2
            }
            var allHeaders = "h1, h2, h3, h4, h5, .headline, .article_title, .post-title, .title, #hn-headline, .inside-head, .instapaper_title";
            var tagName = this.article.element.tagName;
            "DL" !== tagName && "DD" !== tagName || (allHeaders += ", dt");
            var w = this.contentDocument.querySelectorAll(allHeaders);
            w = Array.prototype.slice.call(w, 0);
            const q = 2;
            for (var k = y.location, W = this.article.element, U = 0; q > U; ++U)
                W.parentElement && (W = W.parentElement);
            for (var H = W.getElementsByTagName("a"), U = 0, V = H.length; V > U; ++U) {
                var z = H[U];
                if (z.offsetTop > this.articleNode().offsetTop + i)
                    break;
                if (t(z, k) && "#" !== z.getAttribute("href")) {
                    w.push(z);
                    break
                }
            }
            for (var G, j = w.map(trimmedInnerTextIgnoringTextTransform), Y = w.length, X = 0, K = [], J = [], Q = [], $ = [], Z = [], ee = [], te = [], U = 0; Y > U; ++U) {
                var ne = w[U],
                    ie = j[U],
                    re = stringSimilarity(A, ie);
                if (b) {
                    var ae = stringSimilarity(b, ie);
                    re += ae, ae > StringSimilarityToDeclareStringsNearlyIdentical && J.push(ne)
                }
                if (x) {
                    var le = stringSimilarity(x, ie);
                    re += le, le > StringSimilarityToDeclareStringsNearlyIdentical && Q.push(ne)
                }
                if (S) {
                    var oe = stringSimilarity(S, ie);
                    re += oe, oe > StringSimilarityToDeclareStringsNearlyIdentical && $.push(ne)
                }
                if (T) {
                    var se = stringSimilarity(T, ie);
                    re += se, se > StringSimilarityToDeclareStringsNearlyIdentical && Z.push(ne)
                }
                if (sailthruHeadline) {
                    var ce = stringSimilarity(sailthruHeadline, ie);
                    re += ce, ce > StringSimilarityToDeclareStringsNearlyIdentical && ee.push(ne)
                }
                if (R) {
                    var me = stringSimilarity(R, ie);
                    re += me, me > StringSimilarityToDeclareStringsNearlyIdentical && te.push(ne)
                }
                re === X ? K.push(ne) : re > X && (X = re, K = [ne])
            }
            if (1 === J.length ? (G = J[0], G.headerText = trimmedInnerTextIgnoringTextTransform(G)) : 1 === Q.length ? (G = Q[0], G.headerText = trimmedInnerTextIgnoringTextTransform(G)) : 1 === $.length ? (G = $[0], G.headerText = trimmedInnerTextIgnoringTextTransform(G)) : 1 === ee.length ? (G = ee[0], G.headerText = trimmedInnerTextIgnoringTextTransform(G)) : 1 === te.length && (G = te[0], G.headerText = trimmedInnerTextIgnoringTextTransform(G)), !G)
                for (var U = 0; Y > U; ++U) {
                    var ne = w[U];
                    if (isElementVisible(ne)) {
                        var de = cachedElementBoundingRect(ne),
                            he = de.left + de.width / 2,
                            ue = de.top + de.height / 2,
                            ge = he - M,
                            fe = ue - F,
                            pe = -1 !== J.indexOf(ne),
                            ve = -1 !== Q.indexOf(ne),
                            Ee = ne.classList.contains("instapaper_title"),
                            Ne = /\bheadline\b/.test(ne.getAttribute("itemprop")),
                            Se = -1 !== $.indexOf(ne),
                            Te = -1 !== Z.indexOf(ne),
                            ye = -1 !== ee.indexOf(ne),
                            Ae = pe || ve || Ee || Ne || Se || Te || ye,
                            be = Math.sqrt(ge * ge + fe * fe),
                            Ce = Ae ? HeaderMaximumDistance : Math.max(HeaderMaximumDistance - be, 0),
                            ie = j[U],
                            xe = ne.getAttribute("property");
                        if (xe) {
                            var De = /dc.title/i.exec(xe);
                            if (De && De[0]) {
                                var Ie = this.contentDocument.querySelectorAll('*[property~="' + De[0] + '"]');
                                if (1 === Ie.length) {
                                    G = ne, G.headerText = ie;
                                    break
                                }
                            }
                        }
                        if (!ByLineBonusRegEx.test(ne.className)) {
                            if (!Ae) {
                                if (be > HeaderMaximumDistance)
                                    continue;
                                if (he < L.left || he > L.right)
                                    continue
                            }
                            if (A && stringsAreNearlyIdentical(ie, A))
                                Ce *= h;
                            else if (isPrefixOrSuffix(ie, A))
                                Ce *= u;
                            else if (ie.length < HeaderMinimumTextLength)
                                continue;
                            if (ie !== C || !b) {
                                var Re = false,
                                    Le = nearestAncestorElementWithTagName(ne, "A");
                                if (Le || (Le = ne.querySelector("a")), Le) {
                                    if ("author" === Le.getAttribute("rel"))
                                        continue;
                                    var Me = Le.host === k.host,
                                        Be = Le.pathname === k.pathname;
                                    if (Me && Be)
                                        Ce *= m;
                                    else {
                                        if (Me && nearestAncestorElementWithTagName(ne, "LI"))
                                            continue;
                                        Ce *= d, Re = true
                                    }
                                }
                                var Fe = fontSizeFromComputedStyle(getComputedStyle(ne));
                                Re || (Ce *= Fe / BaseFontSize), Ce *= 1 + TitleCandidateDepthScoreMultiplier * elementDepth(ne);
                                var Oe = parseInt(this.contentTextStyle().fontSize);
                                parseInt(Fe) > Oe * HeaderFontSizeBonusMinimumRatio && (Ce *= HeaderFontSizeBonusMultiplier), (HeaderBonusRegEx.test(ne.className) || HeaderBonusRegEx.test(ne.getAttribute("id"))) && (Ce *= s);
                                var Pe = ne.parentElement;
                                Pe && (HeaderBonusRegEx.test(Pe.className) || HeaderBonusRegEx.test(Pe.getAttribute("id"))) && (Ce *= s), -1 !== K.indexOf(ne) && (Ce *= c);
                                for (var _e = this.article.element, we = ne; we && we !== _e; we = we.parentElement)
                                    if (SidebarRegex.test(we.className)) {
                                        Ce *= g;
                                        break
                                    }
                                (!G || Ce > G.headerScore) && (G = ne, G.headerScore = Ce, G.headerText = ie)
                            }
                        }
                    }
                }
            if (G && domDistance(G, this.articleNode(), 10) > f && parseInt(getComputedStyle(G).fontSize) < p * Oe && (G = null), G) {
                this._articleTitleElement = G;
                var qe = G.headerText.trim();
                b && isPrefixOrSuffix(b, qe) ? this._articleTitle = b : A && isPrefixOrSuffix(A, qe) ? this._articleTitle = A : this._articleTitle = qe
            }
            return this._articleTitle || (b && isPrefixOrSuffix(b, A) ? this._articleTitle = b : this._articleTitle = A), this._articleTitle
        }
    },
    contentFromUniqueMetadataSelector: function (e, t) {
        var n = e.querySelectorAll(t);
        if (1 !== n.length)
            return null;
        var i = n[0];
        return i && 2 === i.attributes.length ? i.content : null
    },

    postSubtitle: function () {
        function e(e) {
            return elementIsAHeader(e) ? parseInt(/H(\d)?/.exec(e.tagName)[1]) : NaN
        }

        const t = /author|kicker/i,
            n = /sub(head|title)|description|dec?k/i;
        if (this._articleSubhead)
            return this._articleSubhead;
        var i = this.articleNode();
        if (i) {
            var r = this._articleTitleElement;
            if (r) {
                var a,
                    l = e(r),
                    o = cachedElementBoundingRect(r),
                    s = this.contentDocument.querySelector("meta[property='og:description']");
                if (s)
                    a = s.content;
                else {
                    var c = this.contentDocument.querySelector("meta[name=description]");
                    c && (a = c.content)
                }
                for (var m = [nextNonFloatingVisibleElementSibling(r), nextLeafElementForElement(r)], d = m.length, h = 0; d > h; ++h) {
                    var u = m[h];
                    if (u && u !== i) {
                        var g = u.className;
                        if (!t.test(g)) {
                            var f = false;
                            if (elementIsAHeader(u))
                                if (isNaN(l))
                                    f = true;
                                else {
                                    var p = e(u);
                                    p - 1 === l && (f = true)
                                }
                            if (!f && n.test(g) && (f = true), !f && /\bdescription\b/.test(u.getAttribute("itemprop")) && (f = true), !f && a && a === u.innerText && (f = true), f || "summary" !== u.getAttribute("itemprop") || (f = true), f) {
                                var v;
                                if ("META" === u.tagName) {
                                    var E = u.getAttribute("content");
                                    v = E ? E.trim() : "";
                                    var N = u.nextElementSibling;
                                    if (!N || trimmedInnerTextIgnoringTextTransform(N) !== v)
                                        continue;
                                    u = N
                                } else {
                                    if (cachedElementBoundingRect(u).top < (o.bottom + o.top) / 2)
                                        continue;
                                    v = trimmedInnerTextIgnoringTextTransform(u).trim()
                                }
                                if (v.length) {
                                    this._articleSubheadElement = u, this._articleSubhead = v;
                                    break
                                }
                            }
                        }
                    }
                }
                return this._articleSubhead
            }
        }
    },

    /**
     * 搜寻文章 Meta 信息。
     * 需要先调用 {@link findAuthorAndDate}，再执行本方法获取结果。
     */
    cookedMetadataBlock: function () {
        function e(e) {
            function t(e, i) {
                if (e.nodeType === Node.TEXT_NODE)
                    return void(i === n.Left ? e.textContent = e.textContent.trimLeft() : i === n.Right ? e.textContent = e.textContent.trimRight() : e.textContent = e.textContent.trim());
                if (e.nodeType === Node.ELEMENT_NODE) {
                    var r = e.childNodes,
                        a = r.length;
                    if (0 !== a) {
                        if (1 === a)
                            return void t(r[0], i);
                        i !== n.Right && t(r[0], n.Left), i !== n.Left && t(r[a - 1], n.Right)
                    }
                }
            }

            const n = {
                Left: 1,
                Right: 2,
                Both: 3
            };
            t(e);
        }

        this.updateArticleBylineAndDateElementsIfNecessary();
        var t = this.articleBylineElement(),
            n = this.articleDateElement();
        if (!t && !n)
            return null;
        if (t && n) {
            var i = t.compareDocumentPosition(n);
            i & Node.DOCUMENT_POSITION_CONTAINS && (t = null), i & Node.DOCUMENT_POSITION_CONTAINED_BY && (n = null), t === n && (n = null)
        }
        var r,
            a = this.contentDocument.createElement("div"),
            l = false,
            o = false;
        if (t) {
            var r = this.cleanArticleNode(t, t.cloneNode(true), CleaningType.MetadataContent, false);
            e(r), r.innerText.trim() && (l = true, r.classList.add("byline"))
        }
        if (n) {
            var s = this.cleanArticleNode(n, n.cloneNode(true), CleaningType.MetadataContent, false);
            e(s), s.innerText.trim() && (o = true, s.classList.add("date"))
        }
        if (l && a.appendChild(r), l && o) {
            var c = document.createElement("span");
            c.classList.add("delimiter"), a.appendChild(c)
        }
        return o && a.appendChild(s), a
    },
    articleBylineElement: function () {
        return this._articleBylineElement
    },
    findArticleBylineElement: function () {
        var e = this.findArticleBylineElementWithoutRejection();
        return e && ("FOOTER" === e.tagName || e.closest("figure")) ? null : e
    },
    findArticleBylineElementWithoutRejection: function () {
        function e(e) {
            for (var t = new Set, n = new Set, i = e.length, o = 0; i - 1 > o; ++o) {
                var s = e[o],
                    c = e[o + 1];
                if (isElementVisible(s) && isElementVisible(c)) {
                    var m = s.parentElement;
                    m === c.parentElement && (m.contains(l) || (n.add(s.parentElement), t.add(s), t.add(c)))
                }
            }
            var d = new Set(e);
            n.forEach(function (e) {
                d.add(e)
            }), t.forEach(function (e) {
                d["delete"](e)
            }), e = [], d.forEach(function (t) {
                e.push(t)
            });
            var h,
                u = null;
            i = e.length;
            for (var o = 0; i > o; ++o) {
                var s = e[o];
                if (isElementVisible(s)) {
                    var g = cachedElementBoundingRect(s),
                        f = g.left + g.width / 2,
                        p = g.top + g.height / 2,
                        v = r - f,
                        E = a - p,
                        N = Math.sqrt(v * v + E * E);
                    (!u || h > N) && (u = s, h = N)
                }
            }
            return u
        }

        const t = "[itemprop='author'], a[rel='author']";
        const n = "#meta_content, .rich_media_meta_list, .artinfo, .byline, .article-byline, .entry-meta, .author-name, .byline-dateline, [itemprop='author'], a[rel='author']";
        var i,
            r,
            a,
            l = this._articleSubheadElement || this._articleTitleElement;
        if (l)
            var i = l ? cachedElementBoundingRect(l) : null,
                r = i.left + i.width / 2,
                a = i.top + i.height / 2;
        var o = this.contentFromUniqueMetadataSelector(this.contentDocument, "head meta[name=author]");
        if (o || (o = this.contentFromUniqueMetadataSelector(this.contentDocument, "head meta[property=author]")), !o) {
            var s = this.schemaDotOrgMetadataObject();
            if (s) {
                var c = s.author;
                c && "object" == typeof c && (o = c.name)
            }
        }
        var m = this.article.element,
            d = m.querySelectorAll(n);
        if (1 === d.length)
            return d[0];
        var h = l ? l.nextElementSibling : null;
        if (h) {
            if (h.matches(n) || h.innerText === o || (h = h.querySelector(n)), h) {
                var u = h.querySelector("li");
                if (u) {
                    var g = h.querySelector(n);
                    g && (h = g)
                }
            }
            if (h)
                return h
        }
        for (var f = this.contentDocument.getElementsByTagName("a"), p = 0, v = f.length; v > p; ++p) {
            var E = f[p];
            if (trimmedInnerTextIgnoringTextTransform(E) === o)
                return E
        }
        var N = m.closest("article");
        if (l && N) {
            var S = N.querySelectorAll(t),
                T = e(S);
            if (T)
                return T;
            if (S = N.querySelectorAll(n), T = e(S))
                return T;
        }
        return null
    },
    articleDateElement: function () {
        return this._articleDateElement
    },
    findArticleDateElement: function () {
        function e(e) {
            for (var t = e; t && t !== r; t = t.parentElement)
                if (elementIsCommentBlock(t))
                    return true;
            return false
        }

        function t(t) {
            for (var n, i = null, r = t.length, a = 0; r > a; ++a) {
                var l = t[a];
                if (isElementVisible(l) && !e(l)) {
                    var c = cachedElementBoundingRect(l),
                        m = c.left + c.width / 2,
                        d = c.top + c.height / 2,
                        h = o - m,
                        u = s - d,
                        g = Math.sqrt(h * h + u * u);
                    (!i || n > g) && (i = l, n = g)
                }
            }
            return i
        }

        const n = "time, .dateline, .entry-date, #post-date";
        var i = this._articleSubheadElement || this._articleTitleElement,
            r = this.article.element,
            a = i ? i.nextElementSibling : null;
        if (a && (c = a.querySelectorAll(n), 1 === c.length && (a = c[0])), !a || a.matches(n) || a.querySelector(n) || (a = null), a && a.contains(r) && (a = null), a)
            return a;
        var l,
            o,
            s;
        if (i)
            var l = i ? cachedElementBoundingRect(i) : null,
                o = l.left + l.width / 2,
                s = l.top + l.height / 2;
        var c = r.querySelectorAll(n);
        if (c.length)
            return t(c);
        if (r = r.closest("article")) {
            var c = r.querySelectorAll(n);
            if (c.length)
                return t(c)
        }
        return null
    },
    articleDateElementWithBylineElementHint: function (e) {
        function t(e) {
            return /date/.test(e.className) || /\bdatePublished\b/.test(e.getAttribute("itemprop"))
        }

        var n = e.nextElementSibling;
        if (n && t(n))
            return n;
        var i = nextLeafElementForElement(e);
        return i && t(i) ? i : null
    },
    updateArticleBylineAndDateElementsIfNecessary: function () {
        this._didArticleBylineAndDateElementDetection || (this.findAuthorAndDate(), this._didArticleBylineAndDateElementDetection = true)
    },

    findAuthorAndDate: function () {
        var e = this.findArticleBylineElement(),
            t = this.findArticleDateElement();
        !t && e && (t = this.articleDateElementWithBylineElementHint(e)), this._articleDateElement = t, this._articleBylineElement = e
    },
    articleIsLTR: function () {
        if (!this._articleIsLTR) {
            var e = getComputedStyle(this.article.element);
            this._articleIsLTR = e ? "ltr" === e.direction : true
        }
        return this._articleIsLTR
    },
    findSuggestedCandidate: function () {
        var e = this.suggestedRouteToArticle;
        if (!e || !e.length)
            return null;
        var t,
            n;
        for (n = e.length - 1; n >= 0 && (!e[n].id || !(t = this.contentDocument.getElementById(e[n].id))); --n)
            ;
        for (n++, t || (t = this.contentDocument); n < e.length;) {
            for (var i = e[n], r = t.nodeType === Node.DOCUMENT_NODE ? t.documentElement : t.firstElementChild, a = 1; r && a < i.index; r = r.nextElementSibling)
                this.shouldIgnoreInRouteComputation(r) || a++;
            if (!r)
                return null;
            if (r.tagName !== i.tagName)
                return null;
            if (i.className && r.className !== i.className)
                return null;
            t = r;
            n++;
        }
        return isElementVisible(t) ? new CandidateElement(t, this.contentDocument) : null
    },

    /**
     * 遍历所有节点强制找正文，默认 1000ms 超时，如果需要强制寻找正文，则传入 true。
     * @param forceFindingArticle 为 true 时则忽略超时限制，强制寻找正文。
     */
    findArticleBySearchingAllElements: function (forceFindingArticle) {
        var suggestedCandidate = this.findSuggestedCandidate(),
            candidateElements = this.findCandidateElements();
        if (!candidateElements || !candidateElements.length) {
            console.log("因为没有 candidateElements 而返回 suggestedCandidate");
            return suggestedCandidate;
        }

        if (suggestedCandidate && suggestedCandidate.basicScore() >= ReaderMinimumScore) {
            console.log("因为 suggestedCandidate 得分高于 ReaderMinimumScore 而返回 suggestedCandidate");
            return suggestedCandidate;
        }

        for (var highestScoringElement = this.highestScoringCandidateFromCandidates(candidateElements), r = highestScoringElement.element; r !== this.contentDocument; r = r.parentNode)
            if ("BLOCKQUOTE" === r.tagName) {
                for (var blockquoteParent = r.parentNode, l = candidateElements.length, i = 0; l > i; ++i) {
                    var candidateElement = candidateElements[i];
                    if (candidateElement.element === blockquoteParent) {
                        highestScoringElement = candidateElement;
                        break
                    }
                }
                break
            }
        if (suggestedCandidate && highestScoringElement.finalScore() < ReaderMinimumScore) {
            console.log( "因为 highestScoringElement 得分低于 ReaderMinimumScore 而返回 suggestedCandidate");
            return suggestedCandidate;
        }
        if (!forceFindingArticle) {
            if (highestScoringElement.shouldDisqualifyDueToScoreDensity())
                return null;
            if (highestScoringElement.shouldDisqualifyDueToHorizontalRuleDensity())
                return null;
            if (highestScoringElement.shouldDisqualifyDueToHeaderDensity())
                return null;
            if (highestScoringElement.shouldDisqualifyDueToSimilarElements(candidateElements))
                return null
        }
        return highestScoringElement;
    },
    findExtraArticle: function () {
        if (!this.article)
            return null;
        for (var e = 0, t = this.article.element; 3 > e && t; ++e, t = t.parentNode) {
            var n = this.findExtraArticleCandidateElements(t);
            if (n && n.length)
                for (var i, r = this.sortCandidateElementsInDescendingScoreOrder(n), a = 0; a < r.length && (i = r[a], i && i.basicScore()); a++)
                    if (!i.shouldDisqualifyDueToScoreDensity() && !i.shouldDisqualifyDueToHorizontalRuleDensity() && !(i.shouldDisqualifyDueToHeaderDensity() || cachedElementBoundingRect(i.element).height < PrependedArticleCandidateMinimumHeight && cachedElementBoundingRect(this.article.element).width !== cachedElementBoundingRect(i.element).width)) {
                        var l = contentTextStyleForNode(this.contentDocument, i.element);
                        if (l && l.fontFamily === this.contentTextStyle().fontFamily && l.fontSize === this.contentTextStyle().fontSize && i)
                            return i
                    }
        }
        return null
    },
    highestScoringCandidateFromCandidates: function (candidateElements) {
        var highestScore = 0;
        var highestScoringElement = null;
        var candidateElementsCount = candidateElements.length;

        for (var i = 0; i < candidateElementsCount; ++i) {
            var candidateElement = candidateElements[i];
            var score = candidateElement.basicScore();

            if (score >= highestScore) {
                highestScore = score;
                highestScoringElement = candidateElement;
            }
        }
        return highestScoringElement
    },
    sortCandidateElementsInDescendingScoreOrder: function sortCandidateElementsInDescendingScoreOrder(candidateElements) {
        function sortByScore(candidate1, candidate2) {
            if (candidate1.basicScore() !== candidate2.basicScore())
                return candidate2.basicScore() - candidate1.basicScore();
            return candidate2.depth() - candidate1.depth();
        }

        return candidateElements.sort(sortByScore);
    },
    findCandidateElements: function () {
        const MaximumCandidateDetectionTimeInterval = 1000;
        var findCandidateElementsTimeoutDate = Date.now() + MaximumCandidateDetectionTimeInterval;
        var elements = this.contentDocument.getElementsByTagName("*");
        var candidateElements = [];
        var elementsCount = elements.length;

        for (var i = 0; i < elementsCount; ++i) {
            var l = elements[i];
            if (!CandidateTagNamesToIgnore[l.tagName]) {
                var candidate = CandidateElement.candidateIfElementIsViable(l, this.contentDocument);

                if (candidate) {
                    candidateElements.push(candidate);
                }

                if (Date.now() > findCandidateElementsTimeoutDate) {
                    console.assert(false, "ReaderArticleFinder aborting CandidateElement detection due to timeout");
                    candidateElements = [];
                    break
                }
            }
        }

        var candidateElementsCount = candidateElements.length;

        for (var i = 0; i < candidateElementsCount; ++i)
            candidateElements[i].element.candidateElement = candidateElements[i];

        for (var i = 0; i < candidateElementsCount; ++i) {
            var candidateElement = candidateElements[i];
            if ("BLOCKQUOTE" === candidateElement.element.tagName) {
                var parentCandidateElement = candidateElement.element.parentElement.candidateElement;

                if (parentCandidateElement) {
                    parentCandidateElement.addTextNodesFromCandidateElement(candidateElement);
                }
            }
        }

        for (var i = 0; i < candidateElementsCount; ++i)
            candidateElements[i].element.candidateElement = null;

        return candidateElements;
    },
    findExtraArticleCandidateElements: function (e) {
        if (!this.article)
            return [];
        e || (e = this.article.element);
        for (var t = "preceding-sibling::*/descendant-or-self::*", n = this.contentDocument.evaluate(t, e, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null), i = n.snapshotLength, r = [], a = 0; i > a; ++a) {
            var l = n.snapshotItem(a);
            if (!CandidateTagNamesToIgnore[l.tagName]) {
                var o = CandidateElement.extraArticleCandidateIfElementIsViable(l, this.article, this.contentDocument, true);
                o && r.push(o)
            }
        }
        t = "following-sibling::*/descendant-or-self::*", n = this.contentDocument.evaluate(t, e, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null), i = n.snapshotLength;
        for (var a = 0; i > a; ++a) {
            var l = n.snapshotItem(a);
            if (!CandidateTagNamesToIgnore[l.tagName]) {
                var o = CandidateElement.extraArticleCandidateIfElementIsViable(l, this.article, this.contentDocument, false);
                o && r.push(o)
            }
        }
        return r
    },
    // 页面模版
    isGeneratedBy: function (e) {
        var t = this.contentDocument.head ? this.contentDocument.head.querySelector("meta[name=generator]") : null;
        if (!t)
            return false;
        var n = t.content;
        return n ? e.test(n) : false
    },
    isMediaWikiPage: function () {
        return this.isGeneratedBy(/^MediaWiki /)
    },
    isWordPressSite: function () {
        return this.isGeneratedBy(/^WordPress/)
    },
    nextPageURLString: function () {
        if (!this.article)
            return null;
        if (this.isMediaWikiPage())
            return null;
        var e,
            t = 0,
            n = this.article.element;
        n.parentNode && "inline" === getComputedStyle(n).display && (n = n.parentNode);
        for (var i = n, r = cachedElementBoundingRect(n).bottom + LinkMaxVerticalDistanceFromArticle; isElementNode(i) && cachedElementBoundingRect(i).bottom <= r;)
            i = i.parentNode;
        i === n || i !== this.contentDocument && !isElementNode(i) || (n = i);
        var a = this.contentDocument.evaluate(LinkCandidateXPathQuery, n, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null),
            l = a.snapshotLength;
        if (this.pageNumber <= 2 && !this.prefixWithDateForNextPageURL) {
            var o = this.contentDocument.location.pathname,
                s = o.match(LinkDateRegex);
            s && (s = s[0], this.prefixWithDateForNextPageURL = o.substring(0, o.indexOf(s) + s.length))
        }
        for (var c = 0; l > c; ++c) {
            var m = a.snapshotItem(c),
                d = this.scoreNextPageLinkCandidate(m);
            d > t && (e = m, t = d)
        }
        return e ? e.href : null
    },
    // 给 “下一页” 链接的候选元素打分
    scoreNextPageLinkCandidate: function (e) {
        function t(e, t, n, i) {
            t.substring(0, e.length) === e && (t = t.substring(e.length), e = "");
            var r = t.lastInteger();
            if (isNaN(r))
                return false;
            var a = e ? e.lastInteger() : NaN;
            return (isNaN(a) || a >= MaximumExactIntegralValue) && (a = i), r === a ? n.lastInteger() === a + 1 : r === a + 1
        }

        function n(e) {
            for (var t = {}, n = e.substring(1).split("&"), i = n.length, r = 0; i > r; ++r) {
                var a = n[r],
                    l = a.indexOf("=");
                -
                    1 === l ? t[a] = null : t[a.substring(0, l)] = a.substring(l + 1)
            }
            return t
        }

        var i = this.contentDocument.location;
        if (e.host !== i.host)
            return 0;
        if (e.pathname === i.pathname && e.search === i.search)
            return 0;
        if (-1 !== e.toString().indexOf("#"))
            return 0;
        if (anchorLinksToAttachment(e) || anchorLinksToTagOrCategoryPage(e))
            return 0;
        if (!isElementVisible(e))
            return 0;
        var r = cachedElementBoundingRect(e),
            a = this.articleBoundingRect(),
            l = Math.max(0, Math.max(a.top - (r.top + r.height), r.top - (a.top + a.height)));
        if (r.top < a.top)
            return 0;
        if (l > LinkMaxVerticalDistanceFromArticle)
            return 0;
        var o = Math.max(0, Math.max(a.left - (r.left + r.width), r.left - (a.left + a.width)));
        if (o > 0)
            return 0;
        var s = i.pathname,
            c = e.pathname;
        if (this.prefixWithDateForNextPageURL) {
            if (-1 === e.pathname.indexOf(this.prefixWithDateForNextPageURL))
                return 0;
            s = s.substring(this.prefixWithDateForNextPageURL.length), c = c.substring(this.prefixWithDateForNextPageURL.length)
        }
        var m = c.substring(1).split("/");
        m[m.length - 1] || m.pop();
        var d = m.length,
            h = s.substring(1).split("/"),
            u = false;
        h[h.length - 1] || (u = true, h.pop());
        var g = h.length;
        if (g > d)
            return 0;
        for (var f = 0, p = 0, v = e.textContent, E = 0; d > E; ++E) {
            var N = m[E],
                S = g > E ? h[E] : "";
            if (S !== N) {
                if (g - 2 > E)
                    return 0;
                if (N.length >= S.length) {
                    for (var T = 0; N[N.length - 1 - T] === S[S.length - 1 - T];)
                        T++;
                    T && (N = N.substring(0, N.length - T), S = S.substring(0, S.length - T));
                    var y = N.indexOf(S);
                    -
                        1 !== y && (N = N.substring(y))
                }
                t(S, N, v, this.pageNumber) ? p = Math.pow(LinkNextOrdinalValueBase, E - d + 1) : f++
            }
            if (f > 1)
                return 0
        }
        var A = false;
        if (e.search) {
            linkParameters = n(e.search), referenceParameters = n(i.search);
            for (var b in linkParameters) {
                var C = linkParameters[b],
                    x = b in referenceParameters ? referenceParameters[b] : null;
                if (x !== C)
                    if (null === x && (x = ""), null === C && (C = ""), C.length < x.length)
                        f++;
                    else if (t(x, C, v, this.pageNumber)) {
                        if (LinkURLSearchParameterKeyMatchRegex.test(b)) {
                            if (s.toLowerCase() !== c.toLowerCase())
                                return 0;
                            if (this.isWordPressSite() && u)
                                return 0;
                            A = true
                        }
                        if (LinkURLBadSearchParameterKeyMatchRegex.test(b)) {
                            f++;
                            continue
                        }
                        p = Math.max(p, 1 / LinkNextOrdinalValueBase)
                    } else
                        f++
            }
        }
        if (!p)
            return 0;
        if ((LinkURLPageSlashNumberMatchRegex.test(e.href) || LinkURLSlashDigitEndMatchRegex.test(e.href)) && (A = true), !A && d === g && stringSimilarity(s, c) < LinkMinimumURLSimilarityRatio)
            return 0;
        if (LinkURLArchiveSlashDigitEndMatchRegex.test(e))
            return 0;
        var D = LinkMatchWeight * (Math.pow(LinkMismatchValueBase, -f) + p) + LinkVerticalDistanceFromArticleWeight * l / LinkMaxVerticalDistanceFromArticle;
        A && (D += LinkURLSemanticMatchBonus), "LI" === e.parentNode.tagName && (D += LinkListItemBonus);
        var v = e.innerText;
        return LinkNextMatchRegEx.test(v) && (D += LinkNextMatchBonus), LinkPageMatchRegEx.test(v) && (D += LinkPageMatchBonus), LinkContinueMatchRegEx.test(v) && (D += LinkContinueMatchBonus), D
    },
    elementContainsEnoughTextOfSameStyle: function (e, t, n) {
        const i = 110;
        var r = "BODY" === e.tagName,
            a = r ? 2 : 3,
            l = getVisibleNonWhitespaceTextNodes(e, a, i, r, t),
            o = n / scoreMultiplierForElementTagNameAndAttributes(e) / languageScoreMultiplierForTextNodes(l),
            s = {};

        for (var m = 0; m < l.length; ++m) {
            var d = l[m],
                h = d.length,
                u = d.parentElement,
                g = window.getComputedStyle(u),
                f = g.fontFamily + "|" + g.fontSize,
                p = Math.pow(h, TextNodeLengthPower);
            if (s[f]) {
                if ((s[f] += p) > o)
                    break
            } else
                s[f] = p
        }
        for (var f in s)
            if (s[f] > o)
                return true;
        return false
    },
    openGraphMetadataClaimsPageTypeIsArticle: function () {
        if (!this._openGraphMetadataClaimsPageTypeIsArticle) {
            var e = this.contentDocument.querySelector("head meta[property='og:type']");
            this._openGraphMetadataClaimsPageTypeIsArticle = e && "article" === e.content
        }
        return this._openGraphMetadataClaimsPageTypeIsArticle
    },
    pointsToUseForHitTesting: function () {
        const e = window.innerWidth,
            t = e / 4,
            n = e / 2,
            i = 128,
            r = 320;
        var a = [
            [n, 800],
            [n, 600],
            [t, 800],
            [n, 400],
            [n - i, 1100],
            [r, 700],
            [3 * t, 800],
            [e - r, 700]
        ];
        return this.openGraphMetadataClaimsPageTypeIsArticle() && a.push([n - i, 1400]), a
    },

    /**
     * 尝试通过几个坐标点来定位文章。
     */
    findArticleByVisualExamination: function () {

        var e = new Set;
        var hitTestingPoints = this.pointsToUseForHitTesting();
        var hitTestingPointsCount = hitTestingPoints.length;
        var i = AppleDotComAndSubdomainsRegex.test(this.contentDocument.location.hostname.toLowerCase()) ? 7200 : 1800;

        for (var r = 0; hitTestingPointsCount > r; r++)
            for (var a = hitTestingPoints[r][0], l = hitTestingPoints[r][1], o = elementAtPoint(a, l), s = o; s && !e.has(s); s = s.parentElement) {
                if (VeryPositiveClassNameRegEx.test(s.className))
                    return new CandidateElement(s, this.contentDocument);
                if (!CandidateTagNamesToIgnore[s.tagName]) {
                    var c = s.offsetWidth,
                        m = s.offsetHeight;
                    if (!c && !m) {
                        var d = cachedElementBoundingRect(s);
                        c = d.width, m = d.height
                    }
                    if (!(c < CandidateMinimumWidth || m < CandidateMinimumHeight || c * m < CandidateMinimumArea)) {
                        var h = this.elementContainsEnoughTextOfSameStyle(s, e, i);
                        if (e.add(s), h && !(CandidateElement.candidateElementAdjustedHeight(s) < CandidateMinimumHeight)) {
                            var u = new CandidateElement(s, this.contentDocument);
                            if (u.shouldDisqualifyDueToSimilarElements())
                                return null;
                            if (u.shouldDisqualifyDueToHorizontalRuleDensity())
                                return null;
                            if (u.shouldDisqualifyDueToHeaderDensity())
                                return null;
                            if (!u.shouldDisqualifyForDeepLinking())
                                return u
                        }
                    }
                }
            }
        return null
    },

    /**
     * 根据 HTML Meta 信息找正文。
     * @param findArticleMode 寻找文章的模式，值从 {@link FindArticleMode} 中选取。
     */
    findArticleFromMetadata: function (findArticleMode) {
        var schemaDotOrgArticleContainers = this.contentDocument.querySelectorAll(SchemaDotOrgArticleContainerSelector);
        if (schemaDotOrgArticleContainers.length == 1) {
            var firstSchemaDotOrgArticleContainer = schemaDotOrgArticleContainers[0];
            if (firstSchemaDotOrgArticleContainer.matches('article, *[itemprop=articleBody]')) {
                var elementIsViable = CandidateElement.candidateIfElementIsViable(firstSchemaDotOrgArticleContainer, this.contentDocument, true);
                if (elementIsViable) {
                    return findArticleMode === FindArticleMode.ExistenceOfElement ? true : elementIsViable;
                }
            }
            var r = firstSchemaDotOrgArticleContainer.querySelectorAll('article, *[itemprop=articleBody]'),
                a = elementWithLargestAreaFromElements(r);
            if (a) {
                var i = CandidateElement.candidateIfElementIsViable(a, this.contentDocument, true);
                if (i) {
                    return findArticleMode === FindArticleMode.ExistenceOfElement ? true : i;
                }
            }
            return new CandidateElement(firstSchemaDotOrgArticleContainer, this.contentDocument)
        }
        if (this.openGraphMetadataClaimsPageTypeIsArticle()) {
            var l = this.contentDocument.querySelectorAll("main article"),
                o = elementWithLargestAreaFromElements(l);
            if (o) {
                var i = CandidateElement.candidateIfElementIsViable(o, this.contentDocument, true);
                if (i) {
                    return findArticleMode === FindArticleMode.ExistenceOfElement ? true : i;
                }
            }
            var s = this.contentDocument.querySelectorAll("article");
            if (1 === s.length) {
                var i = CandidateElement.candidateIfElementIsViable(s[0], this.contentDocument, true);
                if (i) {
                    return findArticleMode === FindArticleMode.ExistenceOfElement ? true : i;
                }
            }
        }
        return null
    },

    /**
     * 获取文章正文内容。
     */
    postTextContent: function () {
        if (!this._articleTextContent) {
            this.cookedText();
        }
        return this._articleTextContent;
    },

    unformattedArticleTextContentIncludingMetadata: function () {
        var articleNode = this.articleNode();
        if (articleNode) {
            var t = '',
                postTitle = this.postTitle();
            if (postTitle) {
                t += postTitle + '\n';
            }
            var i = this.postSubtitle();
            if (i) {
                t += i + '\n';
            }
            var r = this.cookedMetadataBlock();
            if (r) {
                t += this.plaintextVersionOfNodeAppendingNewlinesBetweenBlockElements(r) + '\n';
            }
            return t + articleNode.innerText;
        }
    },

    plaintextVersionOfNodeAppendingNewlinesBetweenBlockElements: function (element) {
        var treeWalker = this.contentDocument.createTreeWalker(element, NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT, null),
            result = '';
        treeWalker.currentNode = element;
        while (treeWalker.nextNode()) {
            var currentNode = treeWalker.currentNode;
            if (currentNode.nodeType !== Node.TEXT_NODE) {
                var tagName = currentNode.tagName;
                if (tagName !== 'P' && tagName !== 'DIV') {
                    result += '\n';
                }
            } else {
                result += currentNode.textContent;
            }
        }
        return result;
    },

    pageDescription: function () {
        var nameMetas = this.contentDocument.querySelectorAll('head meta[name]');
        for (var i = 0; i < nameMetas.length; i++) {
            var nameMeta = nameMetas[i];
            if (nameMeta.getAttribute('name').toLowerCase() === 'description') {
                var descriptionInMeta = nameMeta.getAttribute('content');
                if (descriptionInMeta) {
                    return descriptionInMeta.trim();
                }
            }
        }
        return null;
    },

    /**
     * 通过网页 <title /> 标签的内容来提取文章名和网站名。
     * @param titleString 网页中 <title /> 标签的内容。
     */
    articleTitleAndSiteNameFromTitleString: function (titleString) {
        const Separators = [' - ', ' \u2013 ', ' \u2014 ', ': ', ' | ', ' \xbb '],
            stringSimilarityCanBeConsideredSuitable = .6;

        var result,
            maxStringSimilarity,
            host = this.contentDocument.location.host,
            hostWithoutSubdomain = host.replace(/^(www|m)\./, ''),
            hostWithoutTLD = hostWithoutSubdomain.replace(/\.(com|info|net|org|edu)$/, '').toLowerCase();

        for (var i = 0; i < Separators.length; i++) {
            var titleData = titleString.split(Separators[i]);
            if (titleData.length === 2) {
                var dataBeforeSeparator = titleData[0].trim(),
                    dataAfterSeparator = titleData[1].trim(),
                    currentMaxStringSimilarity1 = Math.max(stringSimilarity(dataBeforeSeparator.toLowerCase(), hostWithoutSubdomain), stringSimilarity(dataBeforeSeparator.toLowerCase(), hostWithoutTLD)),
                    currentMaxStringSimilarity2 = Math.max(stringSimilarity(dataAfterSeparator.toLowerCase(), hostWithoutSubdomain), stringSimilarity(dataAfterSeparator.toLowerCase(), hostWithoutTLD)),
                    currentMaxStringSimilarity = Math.max(currentMaxStringSimilarity1, currentMaxStringSimilarity2);
                if (!maxStringSimilarity || currentMaxStringSimilarity > maxStringSimilarity) {
                    maxStringSimilarity = currentMaxStringSimilarity;
                    result = currentMaxStringSimilarity1 > currentMaxStringSimilarity2 ? {
                        siteName: dataBeforeSeparator,
                        postTitle: dataAfterSeparator
                    } : {
                        siteName: dataAfterSeparator,
                        postTitle: dataBeforeSeparator
                    };
                }
            }
        }
        if (result && maxStringSimilarity >= stringSimilarityCanBeConsideredSuitable) {
            return result;
        } else {
            return null;
        }
    },

    /**
     * 获取页面信息，包括标题、预览文字，网站名称，缩略图，是否可以使用阅读模式浏览。
     * @param readerTitleMaxLength 标题的最大长度。
     * @param readerPreviewTextMaxLength 预览文字的最大长度。
     */
    pageMetadata: function (readerTitleMaxLength, readerPreviewTextMaxLength) {
        var title,
            previewText = this.pageDescription(),
            isReaderAvailable = false;

        if (this.cookedText()) {
            title = this.postTitle();
            previewText = previewText || this.postTextContent();
            isReaderAvailable = true;
        } else {
            title = this.contentDocument.title;
            this.contentDocument.body && (previewText = previewText || this.contentDocument.body.innerText);
        }

        var mainImageURL = '',
            pageImageURLFromMetadata = this.pageImageURLFromMetadata();
        if (pageImageURLFromMetadata) {
            mainImageURL = pageImageURLFromMetadata;
        }
        else {
            var mainImageNode = this.mainImageNode();
            mainImageNode && (mainImageURL = mainImageNode.src)
        }

        if (!title) {
            title = userVisibleURLString(this.contentDocument.location.href);
        }
        title = title.trim();
        if (readerTitleMaxLength) {
            title = title.substring(0, readerTitleMaxLength);
        }

        var siteName = this.contentFromUniqueMetadataSelector(this.contentDocument, 'head meta[property="og:site_name"]');
        if (!siteName) {
            var c = this.articleTitleAndSiteNameFromTitleString(this.contentDocument.title);
            if (c && c.postTitle === title) {
                siteName = c.siteName;
            }
        }
        if (!siteName) {
            siteName = '';
        }
        previewText = previewText ? previewText.trim() : '';
        if (readerPreviewTextMaxLength) {
            previewText = previewText.substring(0, readerPreviewTextMaxLength);
        }
        previewText = previewText.replace(/[\s]+/g, ' ');
        var info = {
            title: title,
            previewText: previewText,
            siteName: siteName,
            mainImageURL: mainImageURL,
            isReaderAvailable: isReaderAvailable
        };
        return info;
    },

    readingListItemInformation: function () {
        const ReaderTitleMaxLength = 220,
            ReaderPreviewTextMaxLength = 220;
        return this.pageMetadata(ReaderTitleMaxLength, ReaderPreviewTextMaxLength)
    },

    /**
     * 读取公众号的标题，缩略图 url 和公众号名称。
     */
    getMPInfo: function () {
        var title;
        if (msg_title) {
            title = msg_title;
        }
        else {
            title = document.getElementsByTagName('h1')[0].innerHTML.toString();
        }
        var thumbUrl;
        if (msg_cdn_url) {
            thumbUrl = msg_cdn_url;
        }
        else {
            thumbUrl = document.getElementsByTagName('img')[0].src.toString();
        }
        var account;
        if (nickname) {
            account = nickname;
        }
        if (!title) {
            title = '';
        }
        if (!thumbUrl) {
            thumbUrl = '';
        }
        if (!account) {
            account = '';
        }

        var info = {
            'title': title,
            'thumbUrl': thumbUrl,
            'account': account,
        }
        return info;
    }
};

/**
 * Finder 方法的调用。
 */

var MPReaderModeController = new ReaderArticleFinder(document);

var ReaderArticleMainInfo = {};

