package moai.scroller;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Canvas;
import android.view.MotionEvent;
import android.view.animation.Interpolator;
import android.view.animation.OvershootInterpolator;

/**
 * 恒定滚动间隔的滚动器，无边界
 * 
 * @author ericziyiwu
 * 
 */
public class RowScroller extends MScroller {

	/** 无效索引值 */
//	public static final int INVALID_SCREEN = -1;

	protected FastVelocityTracker mVelocityTracker;
	protected int mFlingVelocity;
	protected int mFlingVelocityX;
	protected int mFlingVelocityY;

	protected RowScrollerListener mListener;
	protected Interpolator mInterpolator;
	protected Interpolator mInterpolatorBak;

	protected int mOldScroll;
	protected float mPaddingFactor = 0.5f;

	protected int mContentUnitSize = 1; // 为了防止除0，初始化为1
	protected int mCurrentScreen;
	protected int mDstScreen;
	protected boolean mIsOvershooting;

	protected int mTouchDownP;
	protected int mTouchDownX;
	protected int mTouchDownY;
	protected int mTouchDownScrollP;
	protected int mLastTouchP;
	protected int mLastTouchX;
	protected int mLastTouchY;

	protected int mScrollingDuration = 1000; // 切屏最大时间限制
	protected int mDecelerateDuration = 500;

	boolean mUseEffectorMaxOvershootPercent = true;

	protected int mMaxOverShootPercent = 49;
	protected int mOverShootPercent;

	public static final float DEFAULT_ACC = 1.5F;
	private float mAccFactor = DEFAULT_ACC;
	/**
	 * 构造滚动器并绑定到一个分屏视图上，注意还要另外调用 {@link #setScreenSize}, {@link #setScreenCount}
	 * 方法。
	 * 
	 * @param context
	 *            可以为null，此时使用默认的touch slop
	 * @param screenGroup
	 */
	public RowScroller(Context context, RowScrollerListener screenGroup) {
		this(context, screenGroup, null);
	}

	/**
	 * 构造滚动器并绑定到一个分屏视图上，注意还要另外调用 {@link #setScreenSize}, {@link #setScreenCount}
	 * 方法。
	 * 
	 * @param context
	 *            可以为null，此时使用默认的touch slop
	 * @param screenGroup
	 * @param tracker
	 *            外部传入的触摸速度检测器，如果为null则内部建立一个
	 */
	public RowScroller(Context context, RowScrollerListener screenGroup,
			FastVelocityTracker tracker) {
		super(context);
		assert screenGroup != null; // 如果为null也就没任何意义了
		mInterpolatorBak = MScroller.DEFAULT_INTERPOLATOR;
		mInterpolator = mInterpolatorBak;
		mListener = screenGroup;
		mVelocityTracker = tracker != null ? tracker : new FastVelocityTracker();
	}

	/**
	 * 设置滚动方向，默认为{@link #HORIZONTAL}
	 * 
	 * @param orientation
	 *            取值为{@link #HORIZONTAL}, {@link #VERTICAL}
	 */
	@Override
	public void setOrientation(int orientation) {
		abortAnimation();
		if (orientation == mOrientation) {
			return;
		}
		// 取消原来方向的偏移量
		mScroll = 0;
		if (orientation == HORIZONTAL) {
			mListener.scrollBy(0, -mListener.getScrollY());
		} else {
			mListener.scrollBy(-mListener.getScrollX(), 0);
		}
		mOrientation = orientation;
		updateSize();
	}

	protected void updateSize() {
		// 取消当前方向的偏移量
		mScroll = 0;
		if (mOrientation == HORIZONTAL) {
			mListener.scrollBy(-mListener.getScrollX(), 0);
		} else {
			mListener.scrollBy(0, -mListener.getScrollY());
		}
	}

	/**
	 * 卷动视图容器
	 * 
	 * @param newScroll
	 *            不检查是否越界
	 */
	@Override
	protected void scrollScreenGroup(int newScroll) {
		mOldScroll = mScroll;
		mScroll = newScroll;
		if (mScroll != mOldScroll) {
			if (mOrientation == HORIZONTAL) {
				mListener.scrollBy(mScroll - mOldScroll, 0);
			} else {
				mListener.scrollBy(0, mScroll - mOldScroll);
			}
			final int oldScreen = mCurrentScreen;
			mCurrentScreen = computeRowIndex(mScroll);
			if (mCurrentScreen != oldScreen) {
				mListener.onFirstRowChanged(mCurrentScreen, oldScreen);
			}
			mListener.onScrollChanged(mScroll, mOldScroll);
		} else {
			super.scrollScreenGroup(newScroll);
		}
	}

	/**
	 * 设置插值器
	 * 
	 * @param interpolator
	 */
	public void setInterpolator(Interpolator interpolator) {
		if (interpolator == null) {
			interpolator = MScroller.DEFAULT_INTERPOLATOR;
		}
		mInterpolator = interpolator;
		mInterpolatorBak = mInterpolator;
	}

	/**
	 * 设置切换一屏需要的最长时间
	 * 
	 * @param duration
	 *            单位为毫秒，默认为1000
	 */
	public void setDuration(int duration) {
		duration = Math.max(1, duration);
		mScrollingDuration = duration;
	}

	/**
	 * 设置当前屏幕位置，不会产生动画。 调用者必须先检查dstScreen是否越界，或者先调用{@link #setScreenCount}使其不越界。
	 * 
	 * @param dstScreen
	 */
	public void setCurrentScreen(int dstScreen) {
		// if(isFinished()){
		// return;
		// }
		abortAnimation();
		mDstScreen = dstScreen; // 不作任何限制，因为监听者可能在调用本方法时还没有添加所有子屏
		if (mDstScreen == 0 && mScroll == 0) {
			final int oldScreen = mCurrentScreen;
			mCurrentScreen = 0;
//			if (mCurrentScreen != oldScreen) {
//				mListener.onScreenChanged(mCurrentScreen, oldScreen);
//			}
		} else {
			scrollScreenGroup(mDstScreen * mContentUnitSize);
//			scrollScreenGroup(mListener.getScreenTopByIndex(mDstScreen));
		}
	}

//	/**
//	 * 直接设置偏移量，同样也可能会引起监听者的onScrollChanged和onScreenChanged的回调 注意在连续调用完本方法后，要调用
//	 * {@link #setCurrentScreen(int)} 或者{@link #gotoScreen(int, int, boolean)}
//	 * 方法来修正位置（以及还原到非滚动状态）
//	 * 
//	 * @param index
//	 *            屏幕的索引，支持浮点数值
//	 */
//	public void setScrollIndex(float index) {
//		if (mState != ON_SCROLL) {
//			onScrollStart();
//		}
//		index = Math.max(-mPaddingFactor, Math.min(index, mScreenCount - 1 + mPaddingFactor));
//		onScroll((int) (index * mScreenSize) - mEndScroll);// getDstScreenSize
//	}

	/**
	 * 从当前位置滚动到指定的屏幕位置，会产生动画
	 * 
	 * @param dstScreen
	 * @param duration
	 */
	protected boolean flingToScreen(int dstScreen, int duration) {
		// 在两端采用默认插值器；在中间采用指定的插值器
		Interpolator interpolator = mInterpolatorBak;
//		duration = mDecelerateDuration;
//		interpolator = MScroller.VISCOUS_FLUID_INTERPOLATOR;
		return gotoScreen(dstScreen, duration, interpolator);
	}

	/**
	 * 从当前位置使用指定的插值器滚动到指定的屏幕位置，会产生动画
	 * 
	 * @param dstScreen
	 * @param duration
	 *            切换的时间
	 * @param interpolator
	 *            如果为null，使用默认的插值器
	 */
	protected boolean gotoScreen(int dstScreen, int duration, Interpolator interpolator) {
		mInterpolator = interpolator != null ? interpolator : MScroller.DEFAULT_INTERPOLATOR;

		mDstScreen = dstScreen;
//		int targetTop = mListener.getScreenTopByIndex(mDstScreen);
//		final int delta = mDstScreen * mScreenSize - mScroll; // getFirstVisiableScreenSize
		final int delta = mDstScreen * mContentUnitSize - mScroll;
//		Log.i("wuziyi", "gotoScreen delta:" + delta);
		if (delta == 0 && getCurrentDepth() == 0) {
			final int oldState = mState;
			if (mState != MScroller.FINISHED) {
				mState = MScroller.FINISHED;
				mListener.onScrollFinish(getDstScreen());
			}
			// 如果没有开始滚动就收到 ACTION_UP 事件，那么返回 false 表示不处理该事件
			return oldState != MScroller.FINISHED;
		}
		if (mFlingVelocity != 0 && mInterpolator != VISCOUS_FLUID_INTERPOLATOR) {
			// 计算一个合理的时间，但是限制最大值，不能太慢
			duration = Math.min(duration, computeFlingDuration(delta, mFlingVelocity));
			mFlingVelocity = 0;
		}
		onFling(mScroll, delta, duration);
		mListener.onFlingStart();
		return true;
	}
	
	/**
	 * 从当前屏幕使用当前的插值器滚动到指定的屏幕位置，会产生动画。 由外部直接调用（如响应Home键时）
	 * 
	 * @param duration
	 *            小于0则自动计算时间
	 * 
	 */
	public void gotoScreen(int dstScreen, int duration, boolean noElastic) {
		mListener.onScrollStart();
		// TODO: 使用一个合理的初速度来求时间
		gotoScreen(dstScreen, duration < 0 ? mScrollingDuration : duration, noElastic
				? MScroller.DEFAULT_INTERPOLATOR
				: mInterpolatorBak);
	}

	public final Interpolator getInterpolator() {
		return mInterpolatorBak;
	}

//	public final int getScreenSize() {
//		return mScreenSize;
//	}

	/**
	 * 获取当前屏（显示面积较大的那屏）的索引
	 * 
	 * @return
	 */
	public final int getCurrentScreen() {
		return mCurrentScreen;
	}

	/**
	 * 获取最终停留的屏幕索引
	 * 
	 * @return
	 */
	public int getDstScreen() {
		return mDstScreen;
	}

//	/**
//	 * 获取当前屏（显示面积较大的那屏）的偏移量
//	 * 
//	 * @return
//	 */
//	public final int getCurrentScreenOffset() {
//		int scroll = mScroll;
//		return mListener.getScreenTopByIndex(mCurrentScreen) - scroll;
//	}

	/**
	 * 计算第一行的索引
	 * 
	 * @param scroll
	 * @return 过了当前屏一半就改变当前屏的索引
	 */
	protected int computeRowIndex(int scroll) {
		int ret = (Math.abs(scroll) + mContentUnitSize / 2) / mContentUnitSize;
		if (scroll < 0) {
			return -ret;
		}
		return ret;
	}

	/**
	 * 响应触摸事件
	 * 
	 * @param event
	 * @param action
	 *            在某些特殊情况下可以强制指定为某一值，但是默认应该为event.getAction()
	 * @return
	 */
	@Override
	public boolean onTouchEvent(MotionEvent event, int action) {
		mLastTouchX = (int) event.getX();
		mLastTouchY = (int) event.getY();
		final int p = mOrientation == HORIZONTAL ? mLastTouchX : mLastTouchY;
		int delta = mLastTouchP - p;
		mLastTouchP = p;

		switch (action) {
			case MotionEvent.ACTION_DOWN : {
				int eventAction = event.getAction() & MotionEvent.ACTION_MASK;
				mCurrentTouchSlop = eventAction == action ? mTouchSlop : 0;
				mVelocityTracker.addMovement(event);
				mTouchDownP = mLastTouchP;
				mTouchDownX = mLastTouchX;
				mTouchDownY = mLastTouchY;
				mTouchDownScrollP = mScroll;
//				mTouchDownScreen = mCurrentScreen;
				if (mState == MScroller.ON_FLING) {
					mState = MScroller.TO_SCROLL;
					mListener.onFlingIntercepted();
				}
			}
				break;
			case MotionEvent.ACTION_MOVE : {
				mVelocityTracker.addMovement(event);
				if (mState != MScroller.ON_SCROLL) {
					if (Math.abs(mLastTouchP - mTouchDownP) >= mCurrentTouchSlop) {
						// 开始拖动
						mTouchDownP = mLastTouchP;
						mTouchDownX = mLastTouchX;
						mTouchDownY = mLastTouchY;
						// delta = 0;
						onScrollStart();
						mListener.onScrollStart();
					}
				}
				if (mState == MScroller.ON_SCROLL) {
					onScroll((int) (delta * mAccFactor));
				}
			}
				break;
			case MotionEvent.ACTION_UP :
			case MotionEvent.ACTION_CANCEL : {
				mVelocityTracker.addMovement(event);
				mVelocityTracker.computeCurrentVelocity(1000, mMaxFlingVelocity);
				mFlingVelocityX = (int) mVelocityTracker.getXVelocity();
				mFlingVelocityY = (int) mVelocityTracker.getYVelocity();
				mFlingVelocity = mOrientation == HORIZONTAL ? mFlingVelocityX : mFlingVelocityY;
				mVelocityTracker.clear();
				if (mState == MScroller.TO_SCROLL) {
					// 在很短促的甩动情况下会没有ACTION_MOVE事件导致没有启动拖动（从而不去创建绘图缓冲或者启动硬件加速），这里保证启动拖动
					onScrollStart();
					mListener.onScrollStart();
				}
				if (mFlingVelocity > mMinFlingVelocity && mTouchDownP <= p) {
					flingToScreen(mListener.getPreviousMonthRow(), mScrollingDuration);
				} else if (mFlingVelocity < -mMinFlingVelocity && mTouchDownP >= p) {
					flingToScreen(mListener.getNextMonthRow(), mScrollingDuration);
				} else {
					mFlingVelocity = mMinFlingVelocity;
					return flingToScreen(computeRowIndex(mScroll), mScrollingDuration);
				}
			}
				break;
			default :
				return false;
		}
		return true;
	}

	public int getFlingVelocityX() {
		return mFlingVelocityX;
	}

	public int getFlingVelocityY() {
		return mFlingVelocityY;
	}

	@SuppressLint("WrongCall")
	@Override
	public boolean onDraw(Canvas canvas) {
		invalidateScroll();
		return super.onDraw(canvas);
	}


	@Override
	public void onScroll(int delta) {
		if (delta == 0) {
			return;
		}
		super.onScroll(delta);
	}

	@Override
	protected void invalidate() {
		mListener.invalidate();
	}

	@Override
	protected float onComputeFlingOffset(float t) {
		t = mInterpolator.getInterpolation(t);
		int scroll;
		scroll = isFlingFinished() ? mEndScroll : mStartScroll + Math.round(t * mDeltaScroll);
		mIsOvershooting = !isFlingFinished() && t > 1;
		scrollScreenGroup(scroll);
		if (isFinished()) {
			mListener.onScrollFinish(getDstScreen());
		}
		return t;
	}

	@Override
	public void abortAnimation() {
		if (mState == ON_FLING) {
			super.abortAnimation();
			onComputeFlingOffset(1);
		}
	}

	public void setEffectorMaxOvershootEnabled(boolean enabled) {
		mUseEffectorMaxOvershootPercent = enabled;
		setOvershootPercent(mMaxOverShootPercent);
	}

	/**
	 * 设置当前使用的插值器过冲的百分比，受{@link #setMaxOvershootPercent}的影响
	 * 
	 * @param percent
	 *            建议值[0, 50]
	 */
	public void setOvershootPercent(int percent) {
		if (!mUseEffectorMaxOvershootPercent && percent != mMaxOverShootPercent) {
			return;
		}
		percent = Math.min(percent, mMaxOverShootPercent);
		if (mOverShootPercent == percent) {
			return;
		}
		mOverShootPercent = percent;
		if (percent <= 0) {
			setInterpolator(DEFAULT_INTERPOLATOR);
		} else {
			final float tension = solveOvershootInterpolatorTension(percent);
			setInterpolator(new OvershootInterpolator(tension));
		}
	}

	/**
	 * 设置插值器过冲的最大百分比
	 * 
	 * @param percent
	 *            建议值[0, 50)
	 */
	public void setMaxOvershootPercent(int percent) {
		mMaxOverShootPercent = Math.max(0, Math.min(percent, 49));
		setOvershootPercent(mMaxOverShootPercent);
	}

//	public float getProgress() {
//		return mScroll * mTotalSizeInv;
//	}

	void recycle() {
	}

//	/**
//	 * 获取当前屏的前一屏
//	 * 
//	 * @return
//	 */
//	public int getPreviousScreen() {
//		return mCurrentScreen - 1;
//	}
//
//	/**
//	 * 获取当前屏的下一屏
//	 * 
//	 * @return
//	 */
//	public int getNextScreen() {
//		return mCurrentScreen + 1;
//	}


	public int getTouchDeltaX() {
		return mLastTouchX - mTouchDownX;
	}

	public int getTouchDeltaY() {
		return mLastTouchY - mTouchDownY;
	}

	/**
	 * 根据当前插值器的设置计算甩动一段距离需要的时间
	 * 
	 * @param change
	 *            甩动的距离，以像素为单位
	 * @param velocity
	 *            甩动的初速度，以像素/秒为单位，必须不能为0
	 * @return 需要的时间，以毫秒为单位
	 */
	protected int computeFlingDuration(int change, int velocity) {
		/*
		 * 令f(x)为插值器函数，则实际运动方程为F(t)=b+f(t/d) * c， （b<=>begin, t<=>time,
		 * d<=>duration, c<=>change<=>end-begin），
		 * 则速度即一阶导数F'(t)=c/d*f'(t/d)，给定v，则d=c*f'(0)/v。
		 * 
		 * 对于n次方的减速曲线插值，f(x)=1-(1-x)^n， 有f'(x)=n(1-x)^(n-1), f'(0)=n，则d=nc/v。
		 * 
		 * 对于过冲插值，f(x)=(k+1)(x-1)^3+k(x-1)^2+1，其中k为张力参数，
		 * 则f'(x)=3(k+1)(x-1)^2+2k(x-1)，f'(0)=k+3。
		 * 
		 * 但是区分插值器类型太麻烦了，采用差分近似方法来求f'(0)，并且支持任意插值器。
		 */
		float diff = mInterpolator.getInterpolation(EPSILON) * ONE_OVER_EPSILON;
		return (int) Math.abs(change * diff * 1000 / velocity);
	}

	/**
	 * 计算OvershootInterpolator的张力tension
	 * 
	 * @param percent
	 *            超出部分的百分比
	 * @return
	 */
	private static float solveOvershootInterpolatorTension(int percent) {
		/*
		 * OvershootInterpolator的计算公式：k为张力>=0，t为时间[0, 1]
		 * f(t)=(t-1)^2*((k+1)*(t-1)+k)+1=(k+1)(t-1)^3+k(t-1)^2+1
		 * 导数f'(t)=3(k+1)(t-1)^2+2k(t-1)^2 令f'(t)==0，解得t=1-2k/(3(k+1))，或t=1（舍去）
		 * 代入f(t)，得max(f(t))=4k^3/(27(k+1)^2)+1
		 * 即最大超出部分为g(k)=max(f(t))-1=4k^3/(27(k+1)^2) 使用Mathematica命令
		 * Solve[4k^3/(27(k+1)^2)==0.1, k]
		 * http://www.wolframalpha.com/input/?i=Solve
		 * [4k^3%2F%2827%28k%2B1%29^2%29%3D%3D0.1%2C+k] 解g(k)=0.1，得k=1.70154
		 * 解g(k)=0.5，得k=4.89486 如果我们指定g，那么通过解g(k)的方程就得到张力k了——
		 * Solve[4k^3/(27(k+1)^2)==g, k]
		 * http://www.wolframalpha.com/input/?i=Solve
		 * [4k^3%2F%2827%28k%2B1%29^2%29%3D%3Dg%2C+k] 部分结果如下： percent = 0
		 * tension=NaN percent = 10 tension=1.7015402 percent = 20
		 * tension=2.5923889 percent = 30 tension=3.3940518 percent = 40
		 * tension=4.155745 percent = 50 tension=4.8948593
		 */

		// if(percent <= 0) return 0; // 注意percent为0的时候最后除0会得到NaN
		// // 直接设张力为0，退化成DecelerateInterpolator(1.5f)
		// float g = percent / 100.0f;
		// float g2 = g * g;
		// float g3 = g * g2;
		// double d = 27 * g3 + 36 * g2 + 8 * Math.sqrt(g3 + g2) + 8 * g;
		// d = Math.pow(d, 1.0f / 3);
		// return (float)(0.75f * d + (729 * g2 + 648 * g) / (108 * d) + 2.25f *
		// g);

		// 用查找表记录percent=0,5,10,...,50的结果，其他percent使用线性插值计算，对<5的时候还是有10%以上误差
		final float[] tension = { 0.0f, 1.1652954f, 1.7015402f, 2.1642938f, 2.5923889f, 3.0f,
				3.3940518f, 3.7784798f, 4.155745f, 4.5274878f, 4.8948593f, };
		percent = Math.max(0, Math.min(percent, 49));
		int i = percent / 5;
		return tension[i] + (tension[i + 1] - tension[i]) * (percent / 5.0f - i);
	}

	/**
	 * <br>功能简述:滑动加速因子
	 * <br>功能详细描述:
	 * <br>注意:
	 * @param isAppDraw
	 */
	public void setAccFactor(float factor) {
		mAccFactor = factor;
	}
	
	public void setContentUnitSize(int size) {
		mContentUnitSize = size;
	}
	
//	public int getFirstVisiableScreenSize() {
//		int firstIndex = mListener.getFirstVisiableScreenIndex(mScroll);
//		int size = mListener.getScreenSizeByIndex(firstIndex);
//		return size;
//	}
	
//	public int getTouchDownScreen() {
//		return mTouchDownScreen;
//	}
	
}
