package com.funshion.statistic;

import java.net.HttpURLConnection;
import java.net.NetworkInterface;
import java.net.URI;
import java.net.URL;
import java.util.UUID;
import android.content.ContentResolver;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.provider.Settings;
import android.text.TextUtils;
import android.util.Log;

public final class FunUtils {
    public static String getMacAddress(Context context) {
        String macAddress = readMacAddress("eth0");
        if (!TextUtils.isEmpty(macAddress)) {
            return macAddress;
        }

        macAddress = readMacAddress("wlan0");
        if (!TextUtils.isEmpty(macAddress)) {
            return macAddress;
        }

        final WifiInfo info = ((WifiManager)context.getSystemService(Context.WIFI_SERVICE)).getConnectionInfo();
        return (info != null ? info.getMacAddress() : "");
    }

    public static String getSID(Context context) {
        final ContentResolver resolver = context.getContentResolver();
        final String name = context.getPackageName() + ".id";

        String sid = null;
        try {
            sid = Settings.System.getString(resolver, name);
        } catch (Exception e) {
        }

        if (TextUtils.isEmpty(sid)) {
            final SharedPreferences prefs = context.getSharedPreferences("ident.xml", Context.MODE_PRIVATE);
            sid = prefs.getString(name, null);
            if (TextUtils.isEmpty(sid)) {
                try {
                    sid = UUID.randomUUID().toString().replace("-", "");
                    if (!Settings.System.putString(resolver, name, sid)) {
                        prefs.edit().putString(name, sid).apply();
                    }
                } catch (Exception e) {
                    prefs.edit().putString(name, sid).apply();
                }
            }
        }

        Log.d(TAG, "sid = " + sid);
        return sid;
    }

    public static void close(AutoCloseable c) {
        if (c != null) {
            try {
                c.close();
            } catch (Exception e) {
            }
        }
    }

    public static void close(HttpURLConnection conn) {
        if (conn != null) {
            conn.disconnect();
        }
    }

    public static boolean isNetworkConnected(Context context) {
        final ConnectivityManager cm = (ConnectivityManager)context.getSystemService(Context.CONNECTIVITY_SERVICE);
        final NetworkInfo info = cm.getActiveNetworkInfo();
        return (info != null && info.isConnected());
    }

    /* package */ static void appendValue(StringBuilder data, String key, Object value) {
        data.append(key).append('=').append(value != null ? value : "").append('&');
    }

    public static void reportStatistics(Context context, SQLiteDatabase db) {
        final Cursor cursor = db.rawQuery("SELECT * FROM statistics", null);
        if (cursor.getCount() > 0) {
            final String macAddress = getMacAddress(context);
            final String sid = getSID(context);
            while (cursor.moveToNext()) {
                reportStatistics(db, cursor, sid, macAddress);
            }
        }

        cursor.close();
    }

    private static String readMacAddress(String ifname) {
        String result = "";
        try {
            final NetworkInterface ni = NetworkInterface.getByName(ifname);
            if (ni != null) {
                final byte[] macAddress = ni.getHardwareAddress();
                if (macAddress != null) {
                    result = String.format("%02x:%02x:%02x:%02x:%02x:%02x", macAddress[0], macAddress[1], macAddress[2], macAddress[3], macAddress[4], macAddress[5]);
                }
            }
        } catch (Exception e) {
        }

        Log.i(TAG, "mac = " + result);
        return result;
    }

    private static void reportStatistics(SQLiteDatabase db, Cursor cursor, String sid, String macAddress) {
        HttpURLConnection conn = null;
        try {
            final StringBuilder path = new StringBuilder(32).append('/').append(cursor.getString(SERVICE)).append('/').append(cursor.getString(FUNC));
            final StringBuilder params = new StringBuilder(cursor.getString(DATA));
            final long date = cursor.getLong(DATE);
            appendValue(params, "ctime", date);
            appendValue(params, "sid", sid);
            appendValue(params, "rom_ver", Build.VERSION.SDK_INT);
            params.append("mac=").append(macAddress);

            final URI uri = new URI("http", "stat.funshion.net", path.toString(), params.toString(), null);
            final String url = uri.toASCIIString();
            conn = (HttpURLConnection)new URL(url).openConnection();
            conn.setInstanceFollowRedirects(false);
            conn.setReadTimeout(60000);
            conn.setConnectTimeout(60000);
            conn.connect();

            final int statusCode = conn.getResponseCode();
            Log.d(TAG, "report statistic status code = " + statusCode);
            if (statusCode == 200) {
                db.delete("statistics", "_date = " + date, null);
            }
        } catch (Exception e) {
            Log.e(TAG, "report statistics error", e);
        } finally {
            FunUtils.close(conn);
        }
    }

    private static final String TAG = "StatisticService";
    private static final int DATE = 0;
    private static final int FUNC = 2;
    private static final int DATA = 3;
    private static final int SERVICE = 1;

    private FunUtils() {
    }
}
